/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.VersionConstraint;

/**
 * RequiredBaseConstraintType provides the type definition for the ResourceConstraint elements used in RequiredBase and 
 * LocalizationBase elements. A required base definition differs from a requirement definition in the limited nature of the 
 * constraints that can be specified. The purpose of constraints within a required base is to identify resource instances that 
 * can be correctly updated or localized by the content element. Only constraints related to the basic identity characteristics 
 * of the resource are allowed.
 */
public interface RequiredBaseConstraint extends Describable, DisplayNameable {
	
	/**
	 * Gets the version constraint.
	 * 
	 * The VersionConstraint element defines the set of versions that can serve as a base for the update. 
	 * 
	 * @return Allowed versions for the required base resource.
	 */
	VersionConstraint getVersionConstraint();
	
	/**
	 * Gets the resource reference.
	 * 
	 * The resourceRef attribute value MUST reference the id of the resource element in Topology to which this constraint refers.
	 * 
	 * @return Reference to the resource representing the required base for an update operation. 
	 */
	String getResourceReference();
	
	/**
	 * Gets the ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @return Constraint identifier scoped to the deployment descriptor.
	 */
	String getID();
	
	/**
	 * Gets the desired result of the required base constraint.
	 * 
	 * The required base constraint is met when the boolean result of comparing the declared name and/or version to the actual name 
	 * and/or version is equal to the boolean value specified in testValue.<br><br>
	 * Because the purpose of a required base constraint is to positively identify one or more resources that can serve as the base 
	 * for an update or localization, there MUST always be one ResourceConstraint that has testValue set to true.<br><br>
	 * Additional ResourceConstraints can be defined with testValue set to false. These constraints identify characteristics of the 
	 * same required base resource that must not be true for that resource to serve as the base.
	 * 
	 * @return Desired result of the required base constraint 
	 */
	Boolean getTestValue();
	
	/**
	 * Gets the name.
	 * 
	 * The Name element provides the name by which the resource is known in the deployment environment. The value of Name is compared 
	 * to resource names found in the deployment environment as part of constraint evaluation.<br><br>
	 * If the resource name is declared in the referenced resource definition, it SHOULD NOT be declared here. If the resource name 
	 * is changed by application of the update, the original name SHOULD be declared here and the updated name SHOULD be declared in 
	 * ResultingResource. The name declared here is always the one that represents the required value for the required base.
	 * 
	 * @return Name of the required base resource as understood in the deployment environment.
	 */
	String getName();
	
	/**
	 * Sets the version constraint.
	 * 
	 * The VersionConstraint element defines the set of versions that can serve as a base for the update. 
	 * 
	 * @param versionConstraint Allowed versions for the required base resource.
	 */
	void setVersionConstraint(VersionConstraint versionConstraint);
	
	/**
	 * Sets the resource reference.
	 * 
	 * The resourceRef attribute value MUST reference the id of the resource element in Topology to which this constraint refers.
	 * 
	 * @param resourceReference Reference to the resource representing the required base for an update operation.
	 * @throws InvalidInputException if the parameter is null 
	 */
	void setResourceReference(String resourceReference);
	
	/**
	 * Sets the ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param id Constraint identifier scoped to the deployment descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setID(String id);
	
	/**
	 * Sets the desired result of the required base constraint.
	 * 
	 * The required base constraint is met when the boolean result of comparing the declared name and/or version to the actual name 
	 * and/or version is equal to the boolean value specified in testValue.<br><br>
	 * Because the purpose of a required base constraint is to positively identify one or more resources that can serve as the base 
	 * for an update or localization, there MUST always be one ResourceConstraint that has testValue set to true.<br><br>
	 * Additional ResourceConstraints can be defined with testValue set to false. These constraints identify characteristics of the 
	 * same required base resource that must not be true for that resource to serve as the base.
	 * 
	 * @param testValue Desired result of the required base constraint 
	 * <ul>**default value=true</ul>
	 * 
	 */
	void setTestValue(Boolean testValue);
	
	/**
	 * Sets the name.
	 * 
	 * The Name element provides the name by which the resource is known in the deployment environment. The value of Name is compared 
	 * to resource names found in the deployment environment as part of constraint evaluation.<br><br>
	 * If the resource name is declared in the referenced resource definition, it SHOULD NOT be declared here. If the resource name 
	 * is changed by application of the update, the original name SHOULD be declared here and the updated name SHOULD be declared in 
	 * ResultingResource. The name declared here is always the one that represents the required value for the required base.
	 * 
	 * @param name Name of the required base resource as understood in the deployment environment.
	 */
	void setName(String name);

	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
