/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLExportOperation;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLIFIdentity;


/**
 * The implementation of the file export operation.  This operation accepts the following
 * arguments in the order listed below:
 * <ol>
 * 	<li>A {@link ISMLDocument[]} containing the document that should be included as part of
 * 	the export operation </li>
 * 	<li>A {@link Map} that stores the aliases - can be null</li>
 * 	<li>A {@link RuleBinding[]} indicating the rule bindings - can be null</li>
 * 	<li>A {@link SMLIFIdentity} indicating the SML-IF identity</li>
 * 	<li>A {@link Boolean} indicating whether the meta-data of the documents should be considered - false by default <li>
 * 	<li>A {@link Boolean} indicating whether the target file should be replaced (false will append to file) - true by default <li>
 * </ol>
 * 
 * @author Ali Mehregani
 */
public class FileExportOperation extends AbstractSMLOperation implements ISMLExportOperation
{
	/**
	 * Constructor
	 */
	public FileExportOperation(ISMLRepository repository)
	{
		super(repository);		
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#getId()
	 */
	public String getId()
	{
		return ISMLExportOperation.ID;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#run()
	 */
	@SuppressWarnings("unchecked")
	public void run() throws RepositoryOperationException
	{	
		// Check to make sure that the required arguments are there
		Object[] arguments = getArguments();
		
		if (arguments.length < 4)
			throw new RepositoryOperationException(RepositoryMessages.operationBadArguments);
		
		arguments[1] = arguments[1] == null ? new Hashtable<String, String[]>() : arguments[1];
		arguments[2] = arguments[2] == null ? new RuleBinding[0] : arguments[2];
		if (!checkType(arguments, new Class[]{ISMLDocument[].class,	Map.class, RuleBinding[].class, SMLIFIdentity.class}))
			throw new RepositoryOperationException(RepositoryMessages.operationBadArguments);
				
		boolean useMetadata = arguments.length > 4 && arguments[4] instanceof Boolean ? ((Boolean)arguments[4]).booleanValue() : false;
		boolean replaceTarget = arguments.length > 5 && arguments[5] instanceof Boolean ? ((Boolean)arguments[5]).booleanValue() : true;
		ExportOperation exportOperation = new ExportOperation(getRepository());
		exportOperation.exportFromFiles(
				(ISMLDocument[])arguments[0], 
				(Map<String, String[]>)arguments[1],
				(RuleBinding[])arguments[2], 
				(ISMLIFIdentity)arguments[3],
				useMetadata,
				replaceTarget);				
	}	
}
