/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.reference;

import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileDocument;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;
import org.w3c.dom.Node;

/**
 * An SML document reference.  This class can be used to retrieve the
 * root DOM node of the referenced SML document.  
 * 
 * @author Ali Mehregani
 * @author John Arwe
 */
public class SMLDocumentReference extends URIReference
{

	private ISMLRepository repository;

	/**
	 * The constructor
	 * 
	 * @param repositor The repository
	 * @param reference The referenced document
	 * @throws URISyntaxException If the reference is incorrect
	 */
	public SMLDocumentReference(ISMLRepository repositor, String reference) throws URISyntaxException
	{		
		super(null, reference);
		this.repository = repositor;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.URIReference#retrieveDocumentDOM()
	 */
	public Node retrieveDocumentDOM() throws URISyntaxException
	{
		String document = getDocumentReference();
		
		/* Return what's stored in the repository */
		String scheme = getReference().getScheme();
		if (scheme == null)
		{
			try
			{
				ISMLDocument[] documents = repository.fetchDocuments(new SMLFileMetadata(null, ISMLMetadata.DOCUMENT_TYPE_NONE, null, new String[]{document}, null));
				if (documents == null || documents.length <= 0)
					documents = repository.fetchDocuments(new SMLFileMetadata(document, ISMLMetadata.DOCUMENT_TYPE_NONE, null, null, null));
				
				return documents.length == 1 && documents[0] instanceof SMLFileDocument ? ((SMLFileDocument)documents[0]).getDOMDocument() : null;
			} 
			catch (RepositoryOperationException e)
			{				
				e.printStackTrace();
				return null;
			}			
		}
		
		/* Otherwise we'll need to download the reference document */ 
		try
		{
			return SMLValidatorUtil.retrieveRemoteDocument(getReference().toString());
		} 
		catch (RemoteRetrievalException e)
		{
			return null;
		}	
	}
	
	/**
	 * @return the base
	 */
	public URI getBase()
	{
		return super.getBase();
	}


	/**
	 * @param base the base to set
	 */
	public void setBase(URI base) throws URISyntaxException
	{
		super.setBase(base);
	}	
}
