/*******************************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.importing.ui;

import java.io.File;
import java.io.IOException;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.repository.operations.OperationEvent;
import org.eclipse.cosmos.rm.internal.smlif.SMLPlugin;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLRepositoryManager;
import org.eclipse.cosmos.rm.internal.smlif.dialogs.ResourceSelectionDialog;
import org.eclipse.cosmos.rm.internal.smlif.export.ui.ExportToSMLIFWizard;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.IProgressListener;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLImportOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;

public class ImportFromSMLIFWizardPage extends WizardPage {
	
	private static final int FILE_OVERWRITE = 0;

	protected IStructuredSelection selection;

	protected Text modelDirText;
	protected Text smlifFileText;
	protected Button importFromWorkspaceButton;
	protected Button importFromFileSystemButton;
	protected Button createFolderStructureButton;
	protected Button overwriteFilesWithoutWarningButton;
	protected String initialDestinationDir;

	public ImportFromSMLIFWizardPage(String pageName, IStructuredSelection selection) {
		super(pageName);
		setTitle(pageName);
		
		setSelection(selection);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#createAdvancedControls(org.eclipse.swt.widgets.Composite)
	 */	
	public void createControl(Composite parent) {
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, ImportFromSMLIFWizard.CONTEXT_ID_SMLIF_IMPORT_VIEW);
		Composite topComposite = new Composite(parent, SWT.NONE); 

		GridLayout topCompositeLayout = new GridLayout();
		topCompositeLayout.numColumns = 1;
		topCompositeLayout.makeColumnsEqualWidth = false;
		
		topComposite.setLayout(topCompositeLayout);
		
		Group pathsGroup = createPathsGroup(topComposite);
		createSourceFileSection(pathsGroup);
		createDestinationLocationSection(pathsGroup);
		createCheckBoxOptions(topComposite);
		
		setControl(topComposite);
		String dirNameSuggestion = (getInitialDestinationDir() == null) ?
				SMLPlugin.getDefault().getPreferenceStore().getString(SMLPlugin.IMPORT_OUTPUT_DIRECTORY) :
				getInitialDestinationDir();
			
		if (dirNameSuggestion == null || dirNameSuggestion.length() <= 0)
		{
			dirNameSuggestion = selection == null || selection.size() == 0 ? 
								dirNameSuggestion = "/":  //$NON-NLS-1$
								selectionPathAsString(); 
		}
		modelDirText.setText(dirNameSuggestion);
		smlifFileText.setText(getInitialSourceFileName());
		updatePageComplete();
		Dialog.applyDialogFont(topComposite);
	}

	private String selectionPathAsString() {
		Object selectedFile = selection.getFirstElement();
		if (selectedFile instanceof IFile) {
			return ((IFile) selectedFile).getParent().getFullPath().toString();
		} else {
			return ((IFileStore) selectedFile).getParent().toString();
		}
	}

	protected String getInitialSourceFileName() {
		if (selection == null || selection.isEmpty()) {
			return ""; //$NON-NLS-1$
		}
		Object selectedFile = selection.getFirstElement();
		if (selectedFile instanceof IResource) {
			return ((IResource) selectedFile).getFullPath().toString();
		} else {
			return ((IFileStore) selectedFile).toString();
		}
	}

	protected void createCheckBoxOptions(Composite topComposite) {
		overwriteFilesWithoutWarningButton = new Button(topComposite, SWT.CHECK);
		overwriteFilesWithoutWarningButton.setText(SMLMessages.importWizardOptionOverwriteFiles);
		overwriteFilesWithoutWarningButton.setSelection(true);
		
		createFolderStructureButton = new Button(topComposite, SWT.CHECK);
		createFolderStructureButton.setText(SMLMessages.importWizardOptionCreateStructure);
		createFolderStructureButton.setSelection(true);
	}

	protected Group createPathsGroup(Composite topComposite) {
		Group pathsGroup = new Group(topComposite, SWT.NONE);
		pathsGroup.setText(SMLMessages.importWizardPathsGroupTitle);
		GridData pathsGroupData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		pathsGroup.setLayoutData(pathsGroupData);

		GridLayout pathsGroupLayout = new GridLayout();
		pathsGroupLayout.numColumns = 3;
		pathsGroupLayout.makeColumnsEqualWidth = false;
		pathsGroup.setLayout(pathsGroupLayout);
		return pathsGroup;
	}

	protected void createDestinationLocationSection(Group pathsGroup) {
		Label modelDirLabel = new Label(pathsGroup, SWT.NULL);
		modelDirLabel.setText(SMLMessages.importWizardLabelDestinationDirectory);
		
		modelDirText = new Text(pathsGroup, SWT.BORDER); //NON-NLS-1 //NON-NLS-2
		modelDirText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(modelDirText.getText().length() > 0);
			}
		});
		modelDirText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Button browseForDirectoryButton = new Button(pathsGroup, SWT.PUSH);
		browseForDirectoryButton.setText(SMLMessages.importWizardBrowseButton);
		browseForDirectoryButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				promptForDestinationDirectory();
			}
		});
	}

	protected void createSourceFileSection(Group pathsGroup) {
		Label smlifFileLabel = new Label(pathsGroup, SWT.NULL);
		smlifFileLabel.setText(SMLMessages.importWizardLabelSourceFile);

		smlifFileText = new Text(pathsGroup, SWT.BORDER);
		smlifFileText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updatePageComplete();
				
				// Initialize the model dir to something more
				// useful when the smlif file is changed
				// since eclipse does not respond well to
				// generating files to the workspace root dir
				if ("/".equals(modelDirText.getText())) { //$NON-NLS-1$
					if (!"".equals(smlifFileText.getText())) { //$NON-NLS-1$
						Path smlifFilePath = new Path(smlifFileText.getText());
						IFile workspaceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(smlifFilePath);
						String calculatedDirName = smlifFilePath.removeLastSegments(1).toString();
						if ((workspaceFile != null) && (checkWorkspaceDirName(calculatedDirName))) {
							// Since the model dir must be in the workspace, we have to check first whether the
							// SMLIF file is from the workspace
							modelDirText.setText(calculatedDirName);
						}
					}
				}
			}
		});
		GridData smlifFileData = new GridData(GridData.FILL_HORIZONTAL);
		smlifFileData.horizontalSpan = 2;
		smlifFileText.setLayoutData(smlifFileData);
		
		createBrowseButtonRow(pathsGroup);
		
	}

	protected void createBrowseButtonRow(Group pathsGroup) {
		Composite browseButtonRow = new Composite(pathsGroup, SWT.NONE);
		GridLayout browseButtonRowLayout = new GridLayout();
		browseButtonRowLayout.numColumns = 2;
		browseButtonRowLayout.marginHeight = 0;
		browseButtonRowLayout.marginWidth = 0;
		browseButtonRowLayout.makeColumnsEqualWidth = true;
		GridData gd = new GridData(GridData.HORIZONTAL_ALIGN_END);
		gd.horizontalSpan = GridData.FILL;
		browseButtonRow.setLayoutData(gd);
		browseButtonRow.setLayout(browseButtonRowLayout);

		Button importFromWorkspaceButton = new Button(browseButtonRow, SWT.PUSH);
		importFromWorkspaceButton.setText(SMLMessages.importWizardBrowseWorkspaceButtonText);
		importFromWorkspaceButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				String selectedSourceFile = promptForFileInWorkspace();
				updateSourceFileText(selectedSourceFile);
			}
		});
		Button importFromFileSystemButton = new Button(browseButtonRow, SWT.PUSH);
		importFromFileSystemButton.setText(SMLMessages.importWizardBrowseFileSystemButtonText);
		importFromFileSystemButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				String selectedSourceFile = promptForFileOnSystem();
				updateSourceFileText(selectedSourceFile);
			}
		});
	}

	public boolean doImport() 
	{
		String targetDirectory = modelDirText.getText();
		SMLPlugin.getDefault().getPreferenceStore().setValue(SMLPlugin.IMPORT_OUTPUT_DIRECTORY, targetDirectory);
		if (!checkSourceFileName()) {
			setErrorMessage(NLS.bind(SMLMessages.importWizardErrorInvalidSourceFile, smlifFileText.getText()));
			return false;
		}
		if (targetDirectory.charAt(0) != '/') {
			modelDirText.setText('/'+targetDirectory);
		}
		
		targetDirectory = modelDirText.getText();
		if (!checkWorkspaceDirName(targetDirectory)) {
			setErrorMessage(NLS.bind(SMLMessages.importWizardErrorInvalidDestinationDirectory, targetDirectory));
			return false;
		}
			
		ISMLRepository repository = null;
		String projectName = SMLRepositoryUtil.firstSegment(targetDirectory, 1);
		try 
		{
			String fileName = SMLValidatorUtil.retrieveAbsolutePath(smlifFileText.getText());
			
			int smlifFileType = SMLValidatorUtil.identifyDocumentType(new File(fileName));
			
			if (smlifFileType != ISMLConstants.TYPE_SMLIF) {
				SMLCommonUtil.openErrorWithDetail(
						SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
						SMLMessages.importErrorTitle, 
						NLS.bind(SMLMessages.importWizardErrorNotSMLIFFile, fileName), 
						null);
				return false;
			}
			
			int projectSuffixIndex = targetDirectory.indexOf(projectName) + projectName.length() + 1;
			String remainingPath = 	targetDirectory.length() > projectSuffixIndex ? 
									targetDirectory.substring(projectSuffixIndex) : 
									""; //$NON-NLS-1$
				
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
			if (!project.exists())
			{
				// Attempt to create the project
				if (createFolderStructureButton.getSelection())
				{
					try
					{
						project.create(new NullProgressMonitor());
						project.open(new NullProgressMonitor());
					} 
					catch (CoreException e)
					{
						SMLCommonUtil.openErrorWithDetail(
								SMLMessages.errorCreatingProjectTitle, 
								SMLMessages.errorCreatingProject, 
								e);
						return false;
					}	
				}
				// Otherwise display an error
				else
				{
					SMLCommonUtil.openErrorWithDetail(
							SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							SMLMessages.errorMissingProjectTitle, 
							SMLMessages.errorMissingProject, null);
					return false;
				}
				
			}
			repository = SMLRepositoryManager.getRepository(ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getLocation().toString());								
			ISMLOperation operation = repository.getOperation(ISMLImportOperation.ID);
			operation.setArguments(new Object[]{fileName, remainingPath, 
					new Boolean(overwriteFilesWithoutWarningButton.getSelection()), 
					new Boolean(createFolderStructureButton.getSelection())});
			operation.addProgressListener(new ProgressListener());
			operation.run();
						
			return true;
		} 
		catch (RepositoryOperationException e)
		{
			SMLCommonUtil.openErrorWithDetail(SMLMessages.importErrorTitle, e.getMessage(), e);
			return false;
		} 
		catch (IOException e)
		{
			SMLCommonUtil.openErrorWithDetail(SMLMessages.importErrorTitle, e.getMessage(), e);
			return false;
		} 
		catch (CoreException e)
		{
			SMLCommonUtil.openErrorWithDetail(SMLMessages.importErrorTitle, e.getMessage(), e);
			return false;
		} 
		
		finally
		{
			if (repository != null)
			{				
				repository.disconnect();				
			}
			
			try
			{
				IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
				if (project != null && project.exists())
					project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
			}
			catch (CoreException e)
			{
				// Ignore this exception
				e.printStackTrace();				
			}
		}
	}

	/**
	 * Answer whether the destination directory entered is a valid
	 * project or folder name.  The directory does not need to
	 * exist.
	 * @param dirName  
	 * @return
	 */
	protected boolean checkWorkspaceDirName(String dirName) {
		boolean okForFolder = ResourcesPlugin.getWorkspace().validatePath(
				dirName, IResource.FOLDER).isOK();
		if (!okForFolder) {
			okForFolder = ResourcesPlugin.getWorkspace().validatePath(
					dirName, IResource.PROJECT).isOK();
		}
		return okForFolder;
	}
	
	protected boolean checkWorkspaceResourceName(String resourceName) {
		return ResourcesPlugin.getWorkspace().validatePath(
				resourceName, IResource.FILE).isOK();
	}


	protected boolean checkSourceFileName() {
		// First check to see if it's a filesystem path
		File srcFile = new File(smlifFileText.getText());
		if (srcFile.exists()) {
			return true;
		}
		// If the file doesn't exist on the file system,
		// check the workspace
		Path srcPath = new Path(smlifFileText.getText());
		return ResourcesPlugin.getWorkspace().getRoot().getFile(srcPath).exists();
	}
	
	public void setSelection(IStructuredSelection selection) {
		this.selection = selection;
	}

	protected void updatePageComplete() {
		setPageComplete((smlifFileText.getText().length() > 0) && (modelDirText.getText().length() > 0));
	}

	protected void promptForDestinationDirectory() {
		ContainerSelectionDialog containerDialog = new ContainerSelectionDialog(getShell(), ResourcesPlugin.getWorkspace().getRoot(), true, SMLMessages.importWizardDestinationDirectoryDialogTitle);
		containerDialog.setMessage(SMLMessages.importWizardDestinationDirectoryDialogMessage);
		if (containerDialog.open() == Window.CANCEL) {
			return;
		}
		String name = ((IPath) containerDialog.getResult()[0]).toString();
		modelDirText.setText(name);
	}

	protected String promptForFileInWorkspace() {
		ResourceSelectionDialog resourceDialog = new ResourceSelectionDialog(getShell(), ISMLConstants.TYPE_SMLIF);
		resourceDialog.setMultiSelection(false);
		resourceDialog.setTitle(SMLMessages.importWizardSourceFileDialogTitle);
		resourceDialog.setMessage(SMLMessages.importWizardFileInstruction);
		IFile[] results = resourceDialog.open();
		if (results.length == 0) {
			return null;
		}
		return results[0].getFullPath().toString();
	}

	protected String promptForFileOnSystem() {
		FileDialog fileDialog = new FileDialog(getShell());
		fileDialog.setText(SMLMessages.importWizardSourceFileDialogTitle);
		if (smlifFileText.getText().length() > 0) {
			fileDialog.setFileName(new File(smlifFileText.getText()).getName());
			fileDialog.setFilterPath(new File(smlifFileText.getText())
					.getParentFile().getAbsolutePath());
		}
		// fileDialog.setFileName(selectedSMLIFFileName);
		// fileDialog.setFilterPath(selectedSMLIFFilePath);
		fileDialog.setFilterExtensions(new String[] { "*.xml", "*" }); //$NON-NLS-1$ //$NON-NLS-2$
		return fileDialog.open();
	}

	protected void updateSourceFileText(String relativePathToFile) {
		if (relativePathToFile == null) {
			return;
		}
		smlifFileText.setText(relativePathToFile);
	}

	public void setInitialDestinationDir(String initialDestinationDir) {
		this.initialDestinationDir = initialDestinationDir;
	}

	public String getInitialDestinationDir() {
		if ((initialDestinationDir == null) || ("".equals(initialDestinationDir))) { //$NON-NLS-1$
			return null;
		}
		return initialDestinationDir;
	}
	
	
	/**
	 * The progress listener used to  
	 * 
	 * @author Ali Mehregani
	 */
	public static class ProgressListener implements IProgressListener
	{
		private boolean displayedWarning;
		
		public void operationInterrupted(OperationEvent event)
		{
			switch (event.type)
			{
				// Missing structure
				case OperationEvent.TYPE_MISSING_STRUCTURE:
					
					// Show a warning if one hasn't already been displayed
					if (!displayedWarning)
					{
						displayedWarning = true;						
						MessageDialog.openWarning(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
								SMLMessages.importMissingStructureTitle, NLS.bind(SMLMessages.importMissingStructure, (String)event.data));											
					}
					
					break;
					
				// The overwrite operation
				case OperationEvent.TYPE_OVERWRITE:
					
					// Ask whether the file should be overwritten 
					MessageDialog dialog = new MessageDialog(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(),
							SMLMessages.importWizardExistingFilePrompterTitle, null, NLS.bind(
									SMLMessages.importWizardExistingFilePrompterMessage, (String)event.data), 
									0, new String[] {
										SMLMessages.importWizardExistingFileOverwriteButtonLabel, 
										SMLMessages.importWizardExistingFileSkipButtonLabel }, 
									FILE_OVERWRITE);
					event.doit = dialog.open() == FILE_OVERWRITE;					
					break;
			}
		}	
	}
}
