/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact;

import java.math.BigInteger;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactArgument;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactOutputVariable;


/**
 * ArtifactType elements define the files, arguments and other information required to perform a particular deployment operation. 
 * Every artifact that can be defined in a content element is an instance of ArtifactType. These are InstallArtifact, UpdateArtifact, 
 * UndoArtifact, UninstallArtifact, RepairArtifact and ConfigArtifact. 
 */
public interface Artifact {
	/**
	 * Gets the content reference.
	 * 
	 * The value MUST be a reference to the id of the primary artifact file defined in a Content element in the package descriptor.<br><br>
	 * Note that it is valid to have no artifact file and drive the operation from arguments alone.<br><br>
	 * When more than one artifact file is needed, contentRef points to the primary artifact file and AdditionalContent.contentRef 
	 * points to any other files used during application of the content element.<br><br>
	 * When resourceRef is defined, contentRef MUST NOT be defined.
	 * 
	 * @return The primary artifact file. Not used if resourceRef is used.
	 */
	String getContentReference();

	/**
	 * Gets the package type.
	 * 
	 * The type attribute identifies the format of the artifact file or files. When there is no artifact file identified, 
	 * type MAY be left undefined. If there is an artifact file or additional files defined, type MUST be defined.
	 * 
	 * @return Type of the primary artifact file.
	 */
	String getPackageType();
	
	/**
	 * Gets the resource reference.
	 * 
	 * Sometimes, artifact files are created during a deployment rather than being contained in the package.
	 * <ul>For example, some install programs create an uninstall program when the software is deployed. The uninstall program 
	 * is the artifact file that is needed by the UninstallArtifact, but is created by, but not contained in, the package. In this 
	 * case, the created artifact file is represented as a ResultingResource.</ul>
	 * An Artifact element that defines resourceRef identifies the resulting resource as its artifact file.<br><br>
	 * When contentRef is defined, resourceRef MUST NOT be defined.<br><br>
	 * The value MUST reference the id of a resource element in Topology.<br><br>
	 * 
	 * @return The resulting resource representing the artifact file. Not used if contentRef is used.
	 */
	String getResourceReference();
	
	/**
	 * Gets the AdditionalContents.
	 * 
	 * AdditionalContent elements MUST be defined when supporting files are needed by the artifact for this operation. 
	 * The content file reference is specified via the contentRef attribute of AdditionalContent.
	 * 
	 * @return A collection of AdditionalContents that are part of the artifact.
	 */
	Collection<AdditionalContent> getAdditionalContents();
	
	/**
	 * Gets the weight.
	 * 
	 * Defining weights for all artifacts and referenced packages in an SDD provides useful information to software that manages 
	 * deployment. The weight of the artifact refers to the relative time taken to deploy the artifact with respect to other 
	 * artifacts and referenced packages in this SDD. 
	 * <ul>For example, if the artifact takes three times as long to deploy as another artifact whose weight is 2, then the 
	 * weight would be 6. The weight numbers have no meaning in isolation and do not describe actual time elapsed. They simply 
	 * provide an estimate of relative time.</ul>
	 * 
	 * @return A BigInteger object that represents the time required to process this artifact relative to all other artifacts in the SDD.
	 */
	BigInteger getWeight();
	
	/**
	 * Get the ArtifactArguments.
	 * 
	 * Inputs to the processing of the artifact MUST be specified by defining an Arguments element. All required inputs MUST be 
	 * included in the arguments list. There are no implied arguments.
	 * <ul>For example, there is no implication that the selected required resource instances will be passed with an InstallArtifact 
	 * on the install operation. If knowledge of those selections is required, instance identifiers should be passed as arguments.</ul>
	 * When one Argument refers to the OutputVariable of another artifact, the output value must be available at the time of processing 
	 * the dependent artifact. 
	 * <ul>For example, an artifact in a content element that is conditioned on the operating system being Linux should not refer to 
	 * the output of an artifact in a content element conditioned on the operating system being Windows .</ul>
	 * A Dependency requirement MUST be defined between the content elements to indicate that the artifact that defines the output 
	 * variable is a pre-requisite of the content element with the dependent artifact.
	 * 
	 * @return A collection of ArtifactArguments that used during processing of the artifact.
	 */
	Collection<ArtifactArgument> getArguments();
	
	/**
	 * Gets the ArtifactOutputVariables.
	 * 
	 * OutputVariables are variables whose values are set by artifact processing.<br><br>
	 * OutputVariables can also be useful in log and trace messages.
	 * 
	 * @return A collection of ArtifactOutputVariables whose values are set during processing of the artifact.
	 */
	Collection<ArtifactOutputVariable> getOutputVariables();


	/**
	 * Sets the content reference.
	 * 
	 * The value MUST be a reference to the id of the primary artifact file defined in a Content element in the package descriptor.<br><br>
	 * Note that it is valid to have no artifact file and drive the operation from arguments alone.<br><br>
	 * When more than one artifact file is needed, contentRef points to the primary artifact file and AdditionalContent.contentRef 
	 * points to any other files used during application of the content element.<br><br>
	 * When resourceRef is defined, contentRef MUST NOT be defined.
	 * 
	 * @param packageContentReference The primary artifact file. Not used if resourceRef is used.
	 */
	void setContentReference(String packageContentReference);
	

	/**
	 * Sets the resource reference.
	 * 
	 * Sometimes, artifact files are created during a deployment rather than being contained in the package.
	 * <ul>For example, some install programs create an uninstall program when the software is deployed. The uninstall program 
	 * is the artifact file that is needed by the UninstallArtifact, but is created by, but not contained in, the package. In this 
	 * case, the created artifact file is represented as a ResultingResource.</ul>
	 * An Artifact element that defines resourceRef identifies the resulting resource as its artifact file.<br><br>
	 * When contentRef is defined, resourceRef MUST NOT be defined.<br><br>
	 * The value MUST reference the id of a resource element in Topology.<br><br>
	 * 
	 * @param resourceReference The resulting resource representing the artifact file. Not used if contentRef is used.
	 */
	void setResourceReference(String resourceReference);
	
	/**
	 * Sets the package type.
	 * 
	 * The type attribute identifies the format of the artifact file or files. When there is no artifact file identified, 
	 * type MAY be left undefined. If there is an artifact file or additional files defined, type MUST be defined.
	 * 
	 * @param packageType Type of the primary artifact file.
	 */
	void setPackageType(String packageType);

	/**
	 * Sets the AdditionalContents.
	 * 
	 * AdditionalContent elements MUST be defined when supporting files are needed by the artifact for this operation. 
	 * The content file reference is specified via the contentRef attribute of AdditionalContent.
	 * 
	 * @param additionalContents A collection of AdditionalContents that are part of the artifact.
	 */
	void setAdditionalContents(Collection<AdditionalContent> additionalContents);
	
	/**
	 * Sets the weight.
	 * 
	 * Defining weights for all artifacts and referenced packages in an SDD provides useful information to software that manages 
	 * deployment. The weight of the artifact refers to the relative time taken to deploy the artifact with respect to other 
	 * artifacts and referenced packages in this SDD. 
	 * <ul>For example, if the artifact takes three times as long to deploy as another artifact whose weight is 2, then the 
	 * weight would be 6. The weight numbers have no meaning in isolation and do not describe actual time elapsed. They simply 
	 * provide an estimate of relative time.</ul>
	 * 
	 * @param weight A BigInteger that represents the time required to process this artifact relative to all other artifacts in the SDD.
	 */
	void setWeight(BigInteger weight);
	
	/**
	 * Sets the arguments.
	 * 
	 * Inputs to the processing of the artifact MUST be specified by defining an Arguments element. All required inputs MUST be 
	 * included in the arguments list. There are no implied arguments.
	 * <ul>For example, there is no implication that the selected required resource instances will be passed with an InstallArtifact 
	 * on the install operation. If knowledge of those selections is required, instance identifiers should be passed as arguments.</ul>
	 * When one Argument refers to the OutputVariable of another artifact, the output value must be available at the time of processing 
	 * the dependent artifact. 
	 * <ul>For example, an artifact in a content element that is conditioned on the operating system being Linux should not refer to 
	 * the output of an artifact in a content element conditioned on the operating system being Windows .</ul>
	 * A Dependency requirement MUST be defined between the content elements to indicate that the artifact that defines the output 
	 * variable is a pre-requisite of the content element with the dependent artifact.
	 * 
	 * @param arguments A collection of ArtifactArguments used during processing of the artifact.
	 */
	void setArguments(Collection<ArtifactArgument> arguments);
	
	/**
	 * Sets the output variables.
	 * 
	 * OutputVariables are variables whose values are set by artifact processing.<br><br>
	 * OutputVariables can also be useful in log and trace messages.
	 *  
	 * @param outputVariables A collection of ArtifactOutputVariable whose values are set during processing of the artifact.
	 */
	void setOutputVariables(Collection<ArtifactOutputVariable> outputVariables);
}
