/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.net.URI;
import java.util.Stack;

import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.AcyclicStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.AcyclicStructure.AcyclicEntry;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.core.FoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * Construct a structure to store elements that
 * have acyclic = true.
 * 
 * @author Ali Mehregani
 * @author John Arwe
 */
public class AcyclicDataBuilder extends AbstractTrackerDataBuilder<AcyclicStructure>
{
	/**
	 * The ID of this data type builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".AcyclicDataTypesList";
			
	/**
	 * Stores names of acyclic elements
	 */
	private Stack<AcyclicEntry> acyclicElements;
	
	/**
	 * A flag used to indicate when to start collecting character
	 * events
	 */
	private boolean storeReference;
	
	/**
	 * A buffer used to collect characters 
	 */
	private StringBuffer characterBuffer;
	
	/**
	 * The structure constructed by this class
	 */
	private AcyclicStructure acyclicTypeStructure;
	
	
	public AcyclicDataBuilder()
	{
		acyclicElements = new Stack<AcyclicEntry>();
		acyclicTypeStructure = new AcyclicStructure();
		super.addEvent(FoundationBuilder.EVENT_CHARACTER);
	}
	
	
	@Override
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{				
		super.startElement(uri, localName, name, attributes);
		if (!acyclicElements.isEmpty())
		{
			if (ISMLConstants.SML_URI.equals(uri) && ISMLConstants.URI_ELEMENT.equals(localName))
			{
				characterBuffer = new StringBuffer();
				storeReference = true;
			}
			
			return;
		}
		
		// Determine if this element is acyclic
		AnnotationResult annotationResult = SMLValidatorUtil.retrieveAnnotation (getPsvi(), ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE, true);
		if (annotationResult == null)
		{
			return;
		}
				
		String acylicValue = annotationResult.getNodes()[0].getNodeValue();
		
		// Acyclic is set to true
		if (SMLValidatorUtil.isTrue(acylicValue) && 
			isAttributeSet(attributes, ISMLConstants.SML_URI, ISMLConstants.REF_ATTRIBUTE) &&
			!isAttributeSet(attributes, ISMLConstants.SML_URI, ISMLConstants.URI_NILREF))
		{
			AcyclicEntry acyclicEntry = new AcyclicEntry();
			acyclicEntry.setURI(uri);
			acyclicEntry.setLocalName(localName);			
			acyclicEntry.setType((XSComplexTypeDefinition)annotationResult.getType());
			acyclicEntry.setLineNumber(getBaseLineNumber() + getLocator().getLineNumber() - 1);			
			acyclicEntry.setAliases(getAliases());
			acyclicEntry.setNodePath(getNodePath());
			acyclicEntry.setDocumentPosition(getOrphanInx());
			acyclicElements.push(acyclicEntry);
		}	
	}
		
	
	/**
	 * Given a set of attributes, check to make sure the attribute with
	 * the namespace URI and local name is set.
	 * 
	 * @param attributes The set of attribute
	 * @param uri The namespace URI
	 * @param localName The local name of the attribute
	 * @return true if the attribute with namespace uri and local name
	 * is set.
	 */
	private boolean isAttributeSet(Attributes attributes, String uri, String localName)
	{
		String value = attributes.getValue(uri, localName);
		Boolean bValue = value == null ? false : SMLValidatorUtil.isTrue(value);
		return bValue;
	}


	@Override
	public void characters(char[] chars, int start, int length) throws SAXException
	{
		if (!storeReference)
		{
			return;
		}
		
		String str = new String (chars, start, length);
		characterBuffer.append(str.trim());
	}
	
	
	@Override
	public void endElement(String uri, String localName, String name) throws SAXException
	{
		URI saveCurrentBaseURI = getCurrentBaseURI();			//	Alias processing may need access to the value, so save the incoming value
		super.endElement(uri, localName, name);
		if (acyclicElements.isEmpty())
		{
			return;
		}
	
		AcyclicEntry topAcyclicEntry = acyclicElements.peek();
		if (storeReference && ISMLConstants.SML_URI.equals(uri) && ISMLConstants.URI_ELEMENT.equals(localName))
		{
			storeReference = false;
		}
		else if (isEqual(uri, topAcyclicEntry.getURI()) &&
			isEqual(localName, topAcyclicEntry.getLocalName()))
		{			
			acyclicElements.pop();
			String candidateRef = characterBuffer == null ? "" : characterBuffer.toString();
			int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();
			String nonRelativeReference = URIReference.validURIReference(candidateRef, saveCurrentBaseURI, 
					getMessageOutputter(), lineNum, SMLValidationMessages.smlReference);
			topAcyclicEntry.setReference(nonRelativeReference);
			acyclicTypeStructure.add (topAcyclicEntry.getType(), topAcyclicEntry);
		}
	}
	
	
	private boolean isEqual(String str1, String str2)
	{
		return str1 == null ? str2 == null : str1.equals(str2);
	}


	public AcyclicStructure getDataStructure()
	{
		return acyclicTypeStructure;
	}
}
