/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPathFunction;
import javax.xml.xpath.XPathFunctionException;

import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This is an implementaion of the dref function specified in the SML
 * specification.  This implementation makes the following assumptions:
 * <ul>
 * 	<li> The function is invoked after parsing the documents to be validated (i.e.
 * 	the data builders have constructed their associated data structure) 
 * 	</li>
 * </ul>
 * This is a singleton class that should be accessed via {@link DerefXPathFunction#instance()} 
 *  
 * @author Ali Mehregani
 * @author John Arwe
 */
public class DerefXPathFunction implements XPathFunction
{
	/**
	 * Arity - The number of arguments for the dref function
	 */
	public static final int ARITY = 1;
	
	/**
	 * The instance of this singleton class
	 */
	private static DerefXPathFunction instance;
	
	/**
	 * The namespace context to be used
	 */
	private NamespaceContext namespaceContext;
	
	/**
	 * The document node currently being processed
	 */
	private Node documentNode;
	
	/**
	 * @see javax.xml.xpath.XPathFunction#evaluate(java.util.List)
	 */
	@SuppressWarnings("unchecked")
	public Object evaluate(List arguments) throws XPathFunctionException
	{
		if (arguments == null || arguments.size() <= 0)
			throw new XPathFunctionException(SMLValidationMessages.errorInvalidArgument);
		
		ArrayNodeList result = new ArrayNodeList();
		
		// Test the first argument and if it happens to be a string, then evaluate it differently
		if (arguments.get(0) instanceof String)
		{
			for (int i = 0, argCount = arguments.size(); i < argCount; i++)
			{
				evaluateExpression(result, SMLValidatorUtil.removeLineBreaks((String)arguments.get(i)), null);
			}
			
			return result;
		}
		
		
		// Otherwise assume that the content of the arguments list is a node list 
		// Retrieve the data structures required
		SMLIFIdentity identity = (SMLIFIdentity) SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
		
		if (identity == null)
			throw new XPathFunctionException(SMLValidationMessages.errorMissingStructure);
		
		NodeList nodeList = null;
		if ((nodeList = (NodeList)arguments.get(0)).getLength() <= 0)
			throw new XPathFunctionException(SMLValidationMessages.errorInvalidArgument);		
		
		// For every node in the node list
		for (int i = 0, nodeCount = nodeList.getLength(); i < nodeCount; i++)
		{
			Node currentNode = nodeList.item(i);
			
			// Attempt to use the PSVI version of the node
			if (documentNode != null)
			{
				int[] indices = findIndex(currentNode);
				Node node = SMLValidatorUtil.findNode(documentNode, indices);
				if (node != null &&
					node.getNamespaceURI().equals(currentNode.getNamespaceURI()) &&
					node.getLocalName().equals(currentNode.getLocalName()))
				{
					currentNode = node;
				}
			}
			
			
			// Determine the uri and name of the node
			String localName = currentNode.getLocalName();
			String uri = currentNode.getNamespaceURI();
			uri = uri == null ? IValidationConstants.EMPTY_STRING : uri;						
			if (localName == null)
				throw new XPathFunctionException(SMLValidationMessages.errorDetermineName);
			
			
			
			// Throw an exception if the node is not a reference
			if (!SMLValidatorUtil.isReference(currentNode))
				throw new XPathFunctionException(NLS.bind(SMLValidationMessages.errorWrongType, localName));
			
			// Check to make sure the reference is not nullified
			if (SMLValidatorUtil.isNullified(currentNode))
			{
				continue;
			}
			
			// Find the reference URI
			NodeList currentChildNodes = currentNode.getChildNodes();
			if (nodeList == null)
			{
				continue;
			}
			
			List<StringBuffer> expressions = new ArrayList<StringBuffer>();
			for (int j = 0, childNodeCount = currentChildNodes.getLength(); j < childNodeCount; j++)
			{
				Node currentChildNode = currentChildNodes.item(j);				
				if (ISMLConstants.SML_URI.equals(currentChildNode.getNamespaceURI()) && ISMLConstants.URI_ELEMENT.equals(currentChildNode.getLocalName()))
				{
					NodeList uriNodes = currentChildNode.getChildNodes();
					StringBuffer expression = new StringBuffer();
					for (int k = 0, uriNodeCount = uriNodes.getLength(); k < uriNodeCount; k++)
					{
						Node uriNode = uriNodes.item(k);
						if (Node.TEXT_NODE == uriNode.getNodeType() && uriNode.getNodeValue() != null)
							expression.append(SMLValidatorUtil.removeLineBreaks(uriNode.getNodeValue(), false));
					}
					
					if (expression.length() > 0)
					{
						expressions.add(expression);
					}
				}
			}
			
			// Make sure each expression resolves to the same element
			ArrayNodeList previousResult = new ArrayNodeList();
			for (int j = 0, expressionCount = expressions.size(); j < expressionCount; j++)
			{
				ArrayNodeList currentResult = new ArrayNodeList();
				evaluateExpression(currentResult, expressions.get(j).toString(), currentNode);
				if (j != 0 && !previousResult.equals(currentResult))
				{
					throw new XPathFunctionException(NLS.bind(SMLValidationMessages.errorInconsistentReference, new String[]{expressions.get(j - 1).toString(), expressions.get(j).toString()}));
				}
				
				previousResult = currentResult;
			}
			
			for (int j = 0, itemCount = previousResult.getLength(); j < itemCount; j++)
			{
				result.add(previousResult.get(j));
			}
		}
		
		return result;
	}


	private void evaluateExpression(ArrayNodeList result, String referenceExp, Node contextNode) throws XPathFunctionException
	{
		Exception exception = null;		
		if (referenceExp == null || referenceExp.length() <= 0)
		{
			return;
		}
		
		try
		{
			Node docNode = documentNode == null ? 
					contextNode == null ? null : contextNode.getOwnerDocument().getDocumentElement() :
					documentNode;
					
//			URI baseURI = null;
//			if (contextNode != null)
//			{	//	TODO Fetch document base URI  
					//	This is harder than it seems.  Likely need to look upward from the context node to the 
					//	document node, merging any explicit xml:base specifications (since each may be a 
					//	relative reference).  If no explicit xml:base is found, use the smlif doc base URI.
					//	Probably need an index to find the smlif doc base URI given a node (or a document node).
//			}
			//	For now, revert to the model base URI since that is what the base code in URIReference did.
			SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
			URIReference uriReference = new URIReference(docNode, referenceExp, identity.convertBaseURI());
			// below commented out code causes NPE
//			if (docNode.getBaseURI() != null) {
//				uriReference.setBase(new URI(docNode.getBaseURI()));
//			}
			Node context = uriReference.retrieveDocumentDOM();
			
			if (context == null)
			{
				return;
			}
			
			String fragmentExpression = uriReference.getFragment();
			if (fragmentExpression == null)
			{
				result.add(context);
				return;
			}
							
			XPointerExpression xpointerExpression = XPointer.compile(fragmentExpression);
			
			// Build the namespace if necessary
			if (getNamespaceContext() == null && documentNode != null && contextNode != null)
			{
				// Make an attempt of building the namespace.  The context node
				// passed in by the XSLT transformation doesn't preserve namespace,
				// we need to use the document node if it's available
				int[] nodeIndex = findIndex (contextNode);				
				setNamespaceContext(SMLValidatorUtil.buildNamespaceContext(documentNode, nodeIndex));
			}
			
			result.add((Node)xpointerExpression.evaluate(getNamespaceContext(), context));				

		} catch (XPointerSyntaxException e)
		{
			exception = e;
		} catch (URISyntaxException e)
		{
			exception = e;
		} catch (BadContextException e)
		{
			exception = e;
		} catch (BadExpressionException e)
		{
			exception = e;
		}
		
		if (exception != null)
			throw new XPathFunctionException(exception); 
	}

	private int[] findIndex(Node node)
	{
		List<Integer> index = new ArrayList<Integer>();
		findIndex (index, node);		
		int[] result = new int[index.size()];
		for (int i = 0; i < result.length; i++)
		{
			result[i] = index.get(i);
		}
		
		return result;
	}

	private void findIndex(List<Integer> indices, Node node)
	{
		Node parentNode = node.getParentNode();
		if (parentNode.getNodeType() != Node.ELEMENT_NODE)
		{
			indices.add(0, 0);
			return;
		}
		
		NodeList children = parentNode.getChildNodes();
		int index = 0;
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			if (node.equals(child))
			{
				indices.add(0, index);
				break;
			}
			
			if (child.getNodeType() == Node.ELEMENT_NODE)
			{
				index++;
			}
		}
		
		findIndex (indices, parentNode);
	}

	/**
	 * Evaluates expression to a node object
	 * 
	 * @param context The context (SML reference element) node
	 * @param SML reference uri expression
	 * @param xml:base URI corresponding to the URI expression
	 * @return return a node object
	 * @throws XPathFunctionException thrown if an error occurs during evaluation
	 * 
	 * Note: 	the xml:base value might have been specified on the sml:uri element (i.e. on 
	 * 			a different element than the SML reference element).  It might be null or a relative reference.
	 * 			Null is possible because valid SML-IF documents always contain a base URI if one is needed,
	 * 			however this code is testing SML-IF validity... null will be the value if a relative URI is used
	 * 			but no xml:base specification exists in the ancestor axis.
	 */
	public Object evaluate(Node context, String reference, URI baseURI) throws XPathFunctionException
	{
		SMLIFIdentity identity = (SMLIFIdentity) SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
		if (identity == null)
			throw new XPathFunctionException(SMLValidationMessages.errorMissingStructure);
		
		Exception exception = null;
		if (reference == null)
		{
			return null;
		}
		
		// Evaluate the reference
		try
		{
			// TODO Find appropriate base URI - using model base URI as existing code did.
			URIReference uriReference = new URIReference(context, reference, identity.convertBaseURI());

			Node documentNode = uriReference.retrieveDocumentDOM();			
			if (documentNode == null)
			{
				return null;
			}
			
			String fragmentExpression = uriReference.getFragment();
			if (fragmentExpression == null)
			{
				return documentNode;
			}
							
			XPointerExpression xpointerExpression = XPointer.compile(fragmentExpression);
			return xpointerExpression.evaluate(getNamespaceContext(), documentNode);				

		} catch (XPointerSyntaxException e)
		{
			exception = e;
		} catch (URISyntaxException e)
		{
			exception = e;
		} catch (BadContextException e)
		{
			exception = e;
		} catch (BadExpressionException e)
		{
			exception = e;
		}
		
		if (exception != null)
			throw new XPathFunctionException(exception); 		
		return null;
	}
	
	
	/**
	 * @return the namespaceContext
	 */
	public NamespaceContext getNamespaceContext()
	{
		return namespaceContext;
	}

	
	/**
	 * @param namespaceContext the namespaceContext to set
	 */
	public void setNamespaceContext(NamespaceContext namespaceContext)
	{
		this.namespaceContext = namespaceContext;
	}
	

	/**
	 * Returns an instance of this class
	 * 
	 * @return The instance of this class
	 */
	public static DerefXPathFunction instance()
	{
		if (instance == null)
			instance = new DerefXPathFunction();
		
		instance.setNamespaceContext(null);
		return instance;
	}
	
	
	private static class ArrayNodeList extends ArrayList<Node> implements NodeList
	{
		/**
		 * Serial version UID
		 */
		private static final long serialVersionUID = 7992097730072474662L;

		public int getLength()
		{
			return super.size();
		}

		public Node item(int index)
		{
			return (Node)super.get(index);
		}	
		
		@Override
		public boolean equals(Object o)
		{
			if (o == this)
				return true;
			
			ArrayNodeList other = null;
			if (!(o instanceof ArrayNodeList) || (other = (ArrayNodeList)o).getLength() != this.getLength())
			{
				return false;
			}
			
			for (int i = 0, size = getLength(); i < size; i++)
			{
				if (!this.item(i).equals(other.item(i)))
					return false;
			}
			
			return true;
		}
	}
	

	/**
	 * @return the documentNode
	 */
	public Node getDocumentNode()
	{
		return documentNode;
	}
	

	/**
	 * @param documentNode the documentNode to set
	 */
	public void setDocumentNode(Node documentNode)
	{
		this.documentNode = documentNode;
	}
}
