/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SDDManagerImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SDD_DASImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDDManager;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDD_DAS;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class SDDManagerImpl_Test extends TestCase {

	protected void setUp() throws Exception {
		super.setUp();		
		
		AllTests.copySchemaFiles();
	}

	protected void tearDown() throws Exception {
		super.tearDown();		
	}
	
	
	/*
	 * Test loading Deployment Descriptors
	 */
	public void testAddDescriptorFromFile_ValidDD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Collection<Document> documents = manager.getDeploymentDescriptors();
		
		assertNotNull(documents.toArray(new Document[0])[0]);
	}
	
	public void testAddDescriptorFromFile_MalformedDD() throws FileNotFoundException, IOException {
		SDDManager manager = new SDDManagerImpl();
		
		try {
			manager.addDescriptorFromFile(new File(AllTests.MALFORMED_DD));
			fail("DD file is malformed but an XMLValidationException was not thrown");
		} catch (XMLValidationException e) {
			// This exception is expected
		}
	}
	
	public void testAddDescriptorFromDocument_ValidDD() throws IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		SDD_DAS sddDas = new SDD_DASImpl();
		
		Document doc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		manager.addDescriptorFromDocument(doc, new File(AllTests.VALID_DD));
		
		Collection<Document> ddDocs = manager.getDeploymentDescriptors();
		assertSame(doc, ddDocs.toArray(new Document[0])[0]);
		
		Collection<Document> allDocs = manager.getAllDescriptors();
		assertSame(doc, allDocs.toArray(new Document[0])[0]);
	}
	

	/*
	 * Test loading Package Descriptors
	 */
	public void testAddDescriptorFromFile_ValidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		Collection<Document> docs = manager.getPackageDescriptors();
		
		assertNotNull(docs.toArray(new Document[0])[0]);
	}

	public void testAddDescriptorFromFile_InvalidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		Collection<Document> docs = manager.getPackageDescriptors();
		
		assertNotNull(docs.toArray(new Document[0])[0]);
	}

	public void testAddDescriptorFromDocument_ValidPD() throws IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		SDD_DAS sddDas = new SDD_DASImpl();
		
		Document doc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_PD));
		manager.addDescriptorFromDocument(doc, new File(AllTests.VALID_PD));
		Collection<Document> docs = manager.getPackageDescriptors();
		
		assertSame(doc, docs.toArray(new Document[0])[0]);
	}
	
	
	/*
	 * Test adding Validation Rules
	 */
	public void testAddValidation_DD() {
		SDDManager manager = new SDDManagerImpl();
		
		for (int i = 0; i < 1000; i++) {
			ValidationRule validationRule = new TrivialValidationRule();
			manager.addValidation(validationRule, SDDManager.DEPLOYMENT_DESCRIPTOR);
		}
		
		Collection<ValidationRule> ddRules = manager.getDeploymentDescriptorRules();
		assertEquals(ddRules.size(), 1000);
		
		Collection<ValidationRule> allRules = manager.getAllValidationRules();
		assertEquals(allRules.size(), 1000);
		
		Collection<ValidationRule> pdRules = manager.getPackageDescriptorRules();
		assertEquals(pdRules.size(), 0);
	}
	
	public void testAddValidation_PD() {
		SDDManager manager = new SDDManagerImpl();
		
		for (int i = 0; i < 1000; i++) {
			ValidationRule validationRule = new TrivialValidationRule();
			manager.addValidation(validationRule, SDDManager.PACKAGE_DESCRIPTOR);
		}
		
		Collection<ValidationRule> pdRules = manager.getPackageDescriptorRules();
		assertEquals(pdRules.size(), 1000);
		
		Collection<ValidationRule> allRules = manager.getAllValidationRules();
		assertEquals(allRules.size(), 1000);
		
		Collection<ValidationRule> ddRules = manager.getDeploymentDescriptorRules();
		assertEquals(ddRules.size(), 0);
	}
	
	
	/*
	 * Test removing Validation Rules
	 */
	public void testRemoveValidation_DD() {
		SDDManager manager = new SDDManagerImpl();
		manager.addValidation(new TrivialValidationRule(), SDDManager.DEPLOYMENT_DESCRIPTOR);
		
		manager.removeValidation(manager.getAllValidationRules().toArray(new ValidationRule[0])[0], SDDManagerImpl.DEPLOYMENT_DESCRIPTOR);
		
		assertEquals(manager.getAllValidationRules().size(), 0);
		assertEquals(manager.getDeploymentDescriptorRules().size(), 0);
	}
	
	public void testRemoveValidation_PD() {
		SDDManager manager = new SDDManagerImpl();
		manager.addValidation(new TrivialValidationRule(), SDDManager.PACKAGE_DESCRIPTOR);
		
		manager.removeValidation(manager.getAllValidationRules().toArray(new ValidationRule[0])[0], SDDManagerImpl.PACKAGE_DESCRIPTOR);
		
		assertEquals(manager.getAllValidationRules().size(), 0);
		assertEquals(manager.getDeploymentDescriptorRules().size(), 0);
	}


	/*
	 * Test removing Descriptors
	 */
	public void testRemoveDescriptors() {
		SDDManager manager = new SDDManagerImpl();
		
		ValidationRule trivialRule = new TrivialValidationRule();
		ValidationRule schemaRule = new TrivialValidationRule();
		
		manager.addValidation(trivialRule, SDDManager.DEPLOYMENT_DESCRIPTOR);
		manager.addValidation(schemaRule, SDDManager.PACKAGE_DESCRIPTOR);
		
		manager.removeValidation(trivialRule, SDDManager.DEPLOYMENT_DESCRIPTOR);
		assertEquals(manager.getAllValidationRules().size(), 1);
		assertEquals(manager.getDeploymentDescriptorRules().size(), 0);
		assertEquals(manager.getPackageDescriptorRules().size(), 1);
		
		manager.removeValidation(schemaRule, SDDManager.PACKAGE_DESCRIPTOR);
		assertEquals(manager.getAllValidationRules().size(), 0);
		assertEquals(manager.getDeploymentDescriptorRules().size(), 0);
		assertEquals(manager.getPackageDescriptorRules().size(), 0);
	}
	
	
	/*
	 * Test validating Descriptors
	 */
	public void testValidateAllDescriptors_Valid() throws XMLValidationException, FileNotFoundException, IOException {
		SDDManager manager = new SDDManagerImpl();
		
		manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		
		Collection<XMLValidationError> validationErrors = manager.validateAllDescriptors();
		
		assertEquals(validationErrors.size(), 0);
	}
	
	public void testValidateAllDescriptors_Invalid() throws XMLValidationException, FileNotFoundException, IOException {
		SDDManager manager = new SDDManagerImpl();
		
		Document ddDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Document pdDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		
		Element resourceElement = ValidatorUtils.findElement(ddDoc, "Resource");
		resourceElement.getParentNode().removeChild(resourceElement);
		Element contentElement = ValidatorUtils.findElement(pdDoc, "Content");
		contentElement.getParentNode().removeChild(contentElement);
		
		Collection<XMLValidationError> validationErrors = manager.validateAllDescriptors();
		
		assertTrue(validationErrors.size() > 0);
	}


	public void testValidateDescriptors_ValidDD_ValidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		Document ddDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Document pdDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		
		Collection<XMLValidationError> ddValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {ddDoc}));
		Collection<XMLValidationError> pdValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {pdDoc}));
		
		assertEquals(ddValidationErrors.size(), 0);
		assertEquals(pdValidationErrors.size(), 0);
	}
	
	public void testValidateDescriptors_ValidDD_InvalidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		Document ddDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Document pdDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_PD));
		
		Element contentElement = ValidatorUtils.findElement(pdDoc, "Contents");
		contentElement.getParentNode().removeChild(contentElement);
		
		Collection<XMLValidationError> ddValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {ddDoc}));
		Collection<XMLValidationError> pdValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {pdDoc}));
		
		assertEquals(ddValidationErrors.size(), 0);
		assertTrue(pdValidationErrors.size() > 0);
	}

	public void testValidateDescriptors_InvalidDD_InvalidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		Document ddDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Document pdDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_PD));

		Element resourceElement = ValidatorUtils.findElement(ddDoc, "Resource");
		resourceElement.getParentNode().removeChild(resourceElement);
		Element contentElement = ValidatorUtils.findElement(pdDoc, "Contents");
		contentElement.getParentNode().removeChild(contentElement);
		
		Collection<XMLValidationError> ddValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {ddDoc}));
		Collection<XMLValidationError> pdValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {pdDoc}));
		
		assertTrue(ddValidationErrors.size() > 0);
		assertTrue(pdValidationErrors.size() > 0);
	}

	public void testValidateDescriptors_InvalidDD_ValidPD() throws FileNotFoundException, IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		Document ddDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_DD));
		Document pdDoc = manager.addDescriptorFromFile(new File(AllTests.VALID_PD));

		Element resourceElement = ValidatorUtils.findElement(ddDoc, "Resource");
		resourceElement.getParentNode().removeChild(resourceElement);
		
		Collection<XMLValidationError> ddValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {ddDoc}));
		Collection<XMLValidationError> pdValidationErrors = manager.validateDescriptors(Arrays.asList(new Document[] {pdDoc}));
		
		assertTrue(ddValidationErrors.size() > 0);
		assertEquals(pdValidationErrors.size(), 0);
	}

		
	/*
	 * Test writing Descriptors
	 */
	public void testWriteAllDescriptors() throws IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		File validDdOutput = File.createTempFile("DeploymentDescriptor", "xml");
		File validPdOutput = File.createTempFile("PackageDescriptor", "xml");
		validDdOutput.deleteOnExit();
		validPdOutput.deleteOnExit();
		
		SDD_DASImpl sddDas = new SDD_DASImpl();
		Document ddDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		Document pdDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_PD));
		
		manager.addDescriptorFromDocument(ddDoc, validDdOutput);
		manager.addDescriptorFromDocument(pdDoc, validPdOutput);
		
		manager.writeAllDescriptors(false);
		
		if (new FileInputStream(validDdOutput).available() == 0) {
			fail("SDDManager failed to write Deployment Descriptor");
		}
		
		if (new FileInputStream(validPdOutput).available() == 0) {
			fail("SDDManager failed to write Package Descriptor");
		}		
	}

	public void testWriteDescriptors_NotInManager() throws IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		File validDdOutput = File.createTempFile("DeploymentDescriptor", "xml");
		File validPdOutput = File.createTempFile("PackageDescriptor", "xml");
		File anotherOutput = File.createTempFile("Empty", "xml");
		validDdOutput.deleteOnExit();
		validPdOutput.deleteOnExit();
		anotherOutput.deleteOnExit();
		
		SDD_DASImpl sddDas = new SDD_DASImpl();
		Document ddDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		Document pdDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_PD));
		Document anotherDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		
		manager.addDescriptorFromDocument(ddDoc, validDdOutput);
		manager.addDescriptorFromDocument(pdDoc, validPdOutput);
		
		manager.writeDescriptors(Arrays.asList(new Document[] {anotherDoc}), false);
		
		if (new FileInputStream(anotherOutput).available() > 0) {
			fail("SDDManager wrote Descriptor that was not added to the manager");
		}
	}

	public void testWriteDescriptors_InManager() throws IOException, XMLValidationException {
		SDDManager manager = new SDDManagerImpl();
		
		File validDdOutput = File.createTempFile("DeploymentDescriptor", "xml");
		File validPdOutput = File.createTempFile("PackageDescriptor", "xml");
		validDdOutput.deleteOnExit();
		validPdOutput.deleteOnExit();
		
		SDD_DASImpl sddDas = new SDD_DASImpl();
		Document ddDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		Document pdDoc = sddDas.loadDocument(new FileInputStream(AllTests.VALID_PD));
		
		manager.addDescriptorFromDocument(ddDoc, validDdOutput);
		manager.addDescriptorFromDocument(pdDoc, validPdOutput);
		
		manager.writeDescriptors(Arrays.asList(new Document[] {ddDoc}), false);

		if (new FileInputStream(validDdOutput).available() == 0) {
			fail("SDDManager failed to write a Descriptor that was in the list of Descriptors to write");
		}

		if (new FileInputStream(validPdOutput).available() > 0) {
			fail("SDDManager wrote Descriptor that was not in the list of Descriptors to write");
		}
	}
}
