/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SchemaValidationRule;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SchematronValidationRule;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.XML_DASImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XML_DAS;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;


public class XML_DASImpl_Test extends TestCase {
	private static final String FSEP = System.getProperty("file.separator");
	
	// Schematron files
	private static final String VALID_DD_SCHEMATRON = AllTests.ARTIFACT_LOCATION + FSEP + "Valid_SchematronRules_DD.sch";
	private static final String INVALID_SCHEMATRON = AllTests.ARTIFACT_LOCATION + FSEP + "Invalid_SchematronRules.sch";

	
	protected void setUp() throws Exception {
		super.setUp();		
	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}
	
	
	/*
	 * Test loading Deployment Descriptors
	 */
	public void testLoadDocument_ValidDD() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();

		Document myDoc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));

		assertNotNull(myDoc);
	}
	
	public void testLoadDocument_InvalidDD() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		Document myDoc = myXmlDas.loadDocument(new FileInputStream(AllTests.INVALID_DD));
		
		assertNotNull(myDoc);
	}
	
	public void testLoadDocument_MalformedDD() throws FileNotFoundException, IOException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		try {
			Document myDoc = myXmlDas.loadDocument(new FileInputStream(AllTests.MALFORMED_DD));
			fail("DD file is malformed but an XMLValidationException was not thrown");
		} catch (XMLValidationException e) {
			// This exception is expected
		}
	}
		
	
	/*
	 * Test loading Package Descriptors
	 */
	public void testLoadDocument_ValidPD() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		Document myDoc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_PD));
		
		assertNotNull(myDoc);
	}
	
	public void testLoadDocument_InvalidPD() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		Document myDoc = myXmlDas.loadDocument(new FileInputStream(AllTests.INVALID_PD));
		
		assertNotNull(myDoc);
	}
	
	
	/*
	 * Test adding Validation Rules
	 */
	public void testAddValidation() {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		for (int i = 0; i < 1000; i++) {
			ValidationRule validationRule = new TrivialValidationRule();
			myXmlDas.addValidation(validationRule);
		}
		
		Collection<ValidationRule> rules = myXmlDas.getValidationRules();
		
		assertEquals(rules.size(), 1000);
	}

	
	/*
	 * Test removing Validation Rules
	 */
	public void testRemoveValidation_EmptyList() {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		ValidationRule validationRule = new TrivialValidationRule();
		myXmlDas.removeValidation(validationRule);
	}
	
	public void testRemoveValidation_NotInList() {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		ValidationRule validationRule = new TrivialValidationRule();
		myXmlDas.addValidation(validationRule);
		
		ValidationRule validationRule2 = new TrivialValidationRule();
		myXmlDas.removeValidation(validationRule2);
	}
	
	public void testRemoveValidation_InList() {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		ValidationRule validationRule = new TrivialValidationRule();
		myXmlDas.addValidation(validationRule);
		myXmlDas.removeValidation(validationRule);
	}
	
	
	/*
	 * Test performing validation
	 */
	public void testValidate_SchemaValid() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		File[] schemaArray = {new File(AllTests.COMMON_SCHEMA), new File(AllTests.DD_SCHEMA)};
		
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(schemaArray));
		
		myXmlDas.addValidation(validationRule);
		
		Document doc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		
		assertEquals(myXmlDas.validate(doc).size(), 0);
	}
	
	public void testValidate_SchemaInvalid() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
		
		File[] schemaArray = new File[] {new File(AllTests.COMMON_SCHEMA), new File(AllTests.DD_SCHEMA)};

		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(schemaArray));

		myXmlDas.addValidation(validationRule);
		Document doc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		doc.getDocumentElement().removeAttribute("descriptorID");
//		doc.getRootObject().setBytes("descriptorID", null);
		
		assertFalse(myXmlDas.validate(doc).size() == 0);
	}

	public void testValidate_SchematronValid() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();

		SchematronValidationRule validationRule = new SchematronValidationRule(new File(VALID_DD_SCHEMATRON));

		myXmlDas.addValidation(validationRule);
		Document doc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		
		assertEquals(myXmlDas.validate(doc).size(), 0);
	}

	public void testValidate_SchematronInvalid() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();
	
		SchematronValidationRule validationRule = new SchematronValidationRule(new File(INVALID_SCHEMATRON));
		
		myXmlDas.addValidation(validationRule);
		Document doc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		
		assertFalse(myXmlDas.validate(doc).size() == 0);
	}

	
	/*
	 * Test saving a changed Document
	 */
	public void testSaveDocumentDocument() throws FileNotFoundException, IOException, XMLValidationException {
		XML_DAS myXmlDas = new XML_DASImpl();

		Document doc = myXmlDas.loadDocument(new FileInputStream(AllTests.VALID_DD));
		// Find the Topology element
		Element topology = ValidatorUtils.findElement(doc, "Topology");
		// Find the first Resource element under Topology
		Element resource = ValidatorUtils.findElement(topology, "Resource");
		// Change the id attribute
		resource.setAttribute("id", "Win32");
		
		myXmlDas.saveDocument(doc, System.out, "UTF-8", false);
	}
}
