/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * A resource is a part of the deployment environment, to which a solution
 * package may be deployed.  A file system is a typical example.
 *
 */
public interface Resource extends Describable {
	
    /**
     * Gets the name.
     * 
     * The resource name is an identifying characteristic of the resource that correlates with a name for the resource in the 
     * deployment environment.<br><br>
     * The type of the Name element, VariableExpressionType, allows the resource name to be expressed as a simple string or in 
     * terms of a user input parameter or other variable. 
     * <ul>An example of a good use of a variable expression in Resource.Name is to make sure that the installation directory is 
     * hosted on a file system that has sufficient space available for deployment. In this example, the file system resource element 
     * would define a HostedResource element for the directory. The Name of the directory would be expressed as a variable expression 
     * that refers to a user input parameter for installation location. Content elements that use the installation directory would 
     * express a requirement on the directory and on the file system with the additional constraint that the file system have a certain 
     * amount of available space (to satisfy the consumption constraints). The fact that both resources are required and that they are 
     * defined with a hostshostedBy relationship in Topology, means that the directory that is used must be the installation directory 
     * and it must be hosted by a file system that meets the consumption constraint for available space.</ul>
     * Only the Variable elements defined in a top level content element can be used to define a resource Name, because these are the 
     * only variables visible within Topology.<br><br>
     * If the name of a resource is changed during deployment, for example, during an update, then the resource name SHOULD NOT be 
     * included in the resource specification. Instead, the pre-update resource name SHOULD be specified in the RequiredBase element 
     * of the installable unit that provides the update, and the post-update name SHOULD be specified in the ResultingResource element 
     * of the same installable unit.
     * 
     * @return The name of the resource as known in the deployment environment.
     */
    String getName();
    
	/**
	 * Gets the resource ID.
	 * 
	 * The id attribute uniquely identifies the resource element within the DeploymentDescriptor. This id value is used by other 
	 * elements in the DeploymentDescriptor to refer to this resource. This value is created by the descriptor author.<br><br>
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages. 
	 * 
	 * @return An identifier of the resource scoped to the descriptor.  
	 */
	String getResourceID();
	
	/**
	 * Gets the resource type.
	 * 
	 * The type attribute defines the class of resource. The value of type correlates with the resource type known for the resource 
	 * in the deployment environment. ResourceTypeNameType restricts type to valid xsd:QNames. The values for type are not defined 
	 * by this specification. Creators of DeploymentDescriptors rely on knowledge of resource types that are understood by supporting 
	 * infrastructure in the target environment. To honor the descriptor authors intent, the deploying infrastructure must be able to 
	 * discover the existence of resources of the types defined in the SDD; the values of the resources properties; and the existence 
	 * and type of resource relationships. The deploying infrastructure also needs to understand how to use the artifact types 
	 * associated with the resource type to create, modify and delete the resource.
	 * 
	 * @return A well-known resource type.
	 */
	String getResourceType();
	
	/**
	 * Gets the hosted resources.
	 * 
	 * A Resource MAY define HostedResource elements. Each HostedResource element is an instance of ResourceType. When both the host 
	 * and the hosted resource participate in a particular solution deployment, the associated resource instances selected for use 
	 * during that deployment must have a hosts relationship.
	 * <ul>For example, a Web application declared to be hosted on a Web server must be hosted on the instance of the Web server that 
	 * is selected for use during the deployment.</ul>
	 * If only the host resource is identified by the DeploymentDescriptors content elements as participating in the solution, then 
	 * there is no assumption that the hosted resource exists.<br><br>
	 * 
	 * @return A collection of resources that participates in the solution and that is hosted by the defining resource.
	 */
	public Collection<Resource> getHostedResources();
	
	/**
	 * Gets the properties.
	 * 
	 * Property elements SHOULD be used when Name alone is not sufficient to identify the resource. The property used represents an 
	 * identifying characteristic of a resource.
	 * 
	 * @return An identity property of the resource.
	 */
	public Properties getProperties();
	
	/**
	 * Sets the name.
	 * 
	 * The resource name is an identifying characteristic of the resource that correlates with a name for the resource in the 
     * deployment environment.<br><br>
     * The type of the Name element, VariableExpressionType, allows the resource name to be expressed as a simple string or in 
     * terms of a user input parameter or other variable. 
     * <ul>An example of a good use of a variable expression in Resource.Name is to make sure that the installation directory is 
     * hosted on a file system that has sufficient space available for deployment. In this example, the file system resource element 
     * would define a HostedResource element for the directory. The Name of the directory would be expressed as a variable expression 
     * that refers to a user input parameter for installation location. Content elements that use the installation directory would 
     * express a requirement on the directory and on the file system with the additional constraint that the file system have a certain 
     * amount of available space (to satisfy the consumption constraints). The fact that both resources are required and that they are 
     * defined with a hostshostedBy relationship in Topology, means that the directory that is used must be the installation directory 
     * and it must be hosted by a file system that meets the consumption constraint for available space.</ul>
     * Only the Variable elements defined in a top level content element can be used to define a resource Name, because these are the 
     * only variables visible within Topology.<br><br>
     * If the name of a resource is changed during deployment, for example, during an update, then the resource name SHOULD NOT be 
     * included in the resource specification. Instead, the pre-update resource name SHOULD be specified in the RequiredBase element 
     * of the installable unit that provides the update, and the post-update name SHOULD be specified in the ResultingResource element 
     * of the same installable unit.
     * 
	 * @param name The name of the resource as known in the deployment environment.
	 */
	void setName(String name);
	
	/**
	 * Sets the resource ID.
	 * 
	 * The id attribute uniquely identifies the resource element within the DeploymentDescriptor. This id value is used by other 
	 * elements in the DeploymentDescriptor to refer to this resource. This value is created by the descriptor author.<br><br>
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages. 
	 * 
	 * @param An identifier of the resource scoped to the descriptor.
	 * @throws InvalidInputException if the parameter is null.
	 */
	void setResourceID(String resourceID) throws InvalidInputException;
	
	
	/**
	 * Sets the resource type.
	 * 
	 * The type attribute defines the class of resource. The value of type correlates with the resource type known for the resource 
	 * in the deployment environment. ResourceTypeNameType restricts type to valid xsd:QNames. The values for type are not defined 
	 * by this specification. Creators of DeploymentDescriptors rely on knowledge of resource types that are understood by supporting 
	 * infrastructure in the target environment. To honor the descriptor authors intent, the deploying infrastructure must be able to 
	 * discover the existence of resources of the types defined in the SDD; the values of the resources properties; and the existence 
	 * and type of resource relationships. The deploying infrastructure also needs to understand how to use the artifact types 
	 * associated with the resource type to create, modify and delete the resource. 
	 * 
	 * @param A well-known resource type.
	 * @throws InvalidInputException if the parameter is null.
	 */
	void setResourceType(String resourceType) throws InvalidInputException;
	
	/**
	 * Sets the hosted resources.
	 * 
	 * A Resource MAY define HostedResource elements. Each HostedResource element is an instance of ResourceType. When both the host 
	 * and the hosted resource participate in a particular solution deployment, the associated resource instances selected for use 
	 * during that deployment must have a hosts relationship.
	 * <ul>For example, a Web application declared to be hosted on a Web server must be hosted on the instance of the Web server that 
	 * is selected for use during the deployment.</ul>
	 * If only the host resource is identified by the DeploymentDescriptors content elements as participating in the solution, then 
	 * there is no assumption that the hosted resource exists.<br><br>
	 * 
	 * @param hostedResources A collection of resources that participates in the solution and that is hosted by the defining resource.
	 * @throws InvalidInputException if the parameter is null.
	 */
	public void setHostedResources(Collection<Resource> hostedResources) throws InvalidInputException;
	
	/**
	 * Sets the properties.
	 * 
	 * Property elements SHOULD be used when Name alone is not sufficient to identify the resource. The property used represents an 
	 * identifying characteristic of a resource.
	 * 
	 * @param props An identity property of the resource.
	 */
	public void setProperties (Properties props);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
