/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;

/**
 * The CompositeUnit element is used to organize content elements within the base or selectable 
 * content hierarchies. It can define any number of InstallableUnits, ConfigurationUnits, ContainedPackages 
 * and other CompositeUnits. Composite units assist in organizing the deployment package. A composite unit 
 * can provide a convenient way to specify variables, requirements, conditions and other information that 
 * applies to every content element defined below the composite unit. Within the selectable content hierarchy, 
 * composite units can be used to group content elements that are selected by feature sets or groups. When 
 * a feature containing a composite unit is selected, all its child content elements are selected by 
 * association. Organization of content within a composite unit does not imply any relationships among the 
 * resources that result from deployment of the composite content.
 */
public interface CompositeUnit extends ContentUnit {
    /**
     * Gets the condition.
     * 
     * When the condition defined in the CompositeInstallable is not met for a particular deployment, the CompositeUnit and all the content elements defined below the CompositeUnit are out of scope for that particular deployment
     * 
     * @return A condition that determines if the CompositeUnit and its child content elements are relevant to a particular deployment.
     */
    Condition getCondition();
    /**
     * Gets the requirements.
     * 
     * These are requirements that must be met regardless of what content is selected for deployment and which conditions within the content hierarchy evaluates to true.<br><br>
     * Requirements that apply only to a portion of the content SHOULD be defined at the point in the content hierarchy where they apply.<br><br>
     * All requirements specified on content elements that are in scope for a particular deployment MUST be met. This represents a logical AND of the requirements. Care should be taken by the SDD author to ensure that conflicting requirements cannot be in scope for the same deployment. 
     * 
     * @return Requirements that must be met before successful application of the CompositeInstallable. 
     */
    Collection<Requirement> getRequirements();
    /**
     * Gets the installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair existing software.
     * 
     * @return InstallableUnits that are part of the composite content.
     */
    Collection<InstallableUnit> getInstallableUnits();
    /**
     * Gets the configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the requirements for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @return ConfigurationUnits that are part of the composite content.
     */
    Collection<ConfigurationUnit> getConfigurationUnits();
    /**
     * Gets the composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can define any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist in organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, conditions and other information that applies to every content element defined below the composite unit. Within the selectable content hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a feature containing a composite unit is selected, all its child content elements are selected by association. Organization of content within a composite unit does not imply any relationships among the resources that result from deployment of the composite content. 
     * 
     * @return CompositeUnits that organizes a subset of the composites content.
     */
    Collection<CompositeUnit> getCompositeUnits();
    /**
     * Gets the contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, if necessary, to satisfy resource constraints.
     * 
     * @return ContainedPackages that are part of the composite content.
     */
    Collection<ReferencedPackage> getContainedPackages();
    
    /**
     * Sets the condition.
     * 
     * When the condition defined in the CompositeInstallable is not met for a particular deployment, the CompositeUnit and all the content elements defined below the CompositeUnit are out of scope for that particular deployment
     * 
     * @param condition A condition that determines if the CompositeUnit and its child content elements are relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    /**
     * Sets the requirements.
     * 
     * These are requirements that must be met regardless of what content is selected for deployment and which conditions within the content hierarchy evaluates to true.<br><br>
     * Requirements that apply only to a portion of the content SHOULD be defined at the point in the content hierarchy where they apply.<br><br>
     * All requirements specified on content elements that are in scope for a particular deployment MUST be met. This represents a logical AND of the requirements. Care should be taken by the SDD author to ensure that conflicting requirements cannot be in scope for the same deployment. 
     * 
     * @param requirements Requirements that must be met before successful application of the CompositeInstallable.
     */
    void setRequirements(Collection<Requirement> requirements);
    /**
     * Sets the installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair existing software.
     *      
     * @param installableUnits InstallableUnits that are part of the composite content.
     */
    void setInstallableUnits(Collection<InstallableUnit> installableUnits);
    /**
     * Sets the configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the requirements for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @param configurationUnits ConfigurationUnits that are part of the composite content.
     */
    void setConfigurationUnits(Collection<ConfigurationUnit> configurationUnits);
    /**
     * Sets the composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can define any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist in organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, conditions and other information that applies to every content element defined below the composite unit. Within the selectable content hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a feature containing a composite unit is selected, all its child content elements are selected by association. Organization of content within a composite unit does not imply any relationships among the resources that result from deployment of the composite content. 
     * 
     * @param compositeUnits CompositeUnits that organizes a subset of the composites content.
     */
    void setCompositeUnits(Collection<CompositeUnit> compositeUnits);
    /**
     * Sets the contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, if necessary, to satisfy resource constraints.
     * 
     * @param containedPackages ContainedPackages that are part of the composite content.
     */
    void setContainedPackages(Collection<ReferencedPackage> containedPackages);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
