/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.receiver;

import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ResourceBundle;

import javax.xml.namespace.QName;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.soap.SOAP11Constants;
import org.apache.axiom.soap.SOAP12Constants;
import org.apache.axiom.soap.SOAPEnvelope;
import org.apache.axiom.soap.SOAPFactory;
import org.apache.axiom.soap.SOAPFault;
import org.apache.axiom.soap.SOAPFaultCode;
import org.apache.axiom.soap.SOAPFaultDetail;
import org.apache.axiom.soap.SOAPFaultReason;
import org.apache.axiom.soap.SOAPFaultSubCode;
import org.apache.axiom.soap.SOAPFaultText;
import org.apache.axiom.soap.SOAPFaultValue;
import org.apache.axiom.soap.impl.llom.soap11.SOAP11Factory;
import org.apache.axiom.soap.impl.llom.soap12.SOAP12Factory;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.engine.MessageReceiver;
import org.apache.axis2.i18n.Messages;
import org.apache.axis2.receivers.AbstractInOutMessageReceiver;
import org.apache.axis2.util.XMLUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * This class is an implementation of a message receiver for axis2 web services.  
 * It is intended to use by CMDBf query and registration web services.  
 * This receiver invokes the implementation methods of a web services operation, 
 * and passes the message body to the method as a parameter.
 * It is also responsible to create a CMDBf spec-compliant SOAP fault if exception 
 * is thrown during the operation. 
 *
 */
public class CosmosMessageReceiver extends AbstractInOutMessageReceiver implements MessageReceiver {

    private Method findOperation(AxisOperation op, Class implClass) {
        Method method = (Method)(op.getParameterValue("myMethod"));
        if (method != null) return method;

        String methodName = op.getName().getLocalPart();

        try {
            // Looking for a method of the form "OMElement method(OMElement)"
            method = implClass.getMethod(methodName, new Class [] { OMElement.class });
            if (method.getReturnType().equals(OMElement.class)) {
                try {
                    op.addParameter("myMethod", method);
                } catch (AxisFault axisFault) {
                    // Do nothing here
                }
                return method;
            }
        } catch (NoSuchMethodException e) {
            // Fault through
        }

        return null;
    }

    /**
     * Invokes the business logic invocation on the service implementation class
     *
     * @param msgContext    the incoming message context
     * @param newmsgContext the response message context
     * @throws AxisFault on invalid method (wrong signature) or behavior (return null)
     */
    public void invokeBusinessLogic(MessageContext msgContext, MessageContext newmsgContext)
            throws AxisFault {
    	
    	OMElement request = msgContext.getEnvelope().getBody().getFirstElement();
    	
    	// implementation class of the service
    	Class implClass = null;
    	// the receiver will log on behave of the service implementation class
    	Logger logger = null;

        try {

            // get the implementation class for the Web Service
            Object obj = getTheImplementationObject(msgContext);

            // find the WebService method
            implClass = obj.getClass();
            AxisOperation opDesc = msgContext.getAxisOperation();
            Method method = findOperation(opDesc, implClass);
            
            // log on behave of the implementation class
        	if (implClass != null) {
        		logger = Logger.getLogger(implClass);
        	} else {
        		// in an highly unlikely situation when the implementation class is not found,
        		// get logger of the receiver class.
        		logger = Logger.getLogger(getClass());
        	}

        	// print the incoming SOAP message as a debug message
        	logger.debug("Incoming SOAP message: \n" + msgContext.getEnvelope());

            if (method == null) {
                throw new AxisFault(Messages.getMessage("methodDoesNotExistInOut",
                                                        opDesc.getName().toString()));
            }

            OMElement result = (OMElement) method.invoke(
                    obj, new Object[]{request});
            SOAPFactory fac = getSOAPFactory(msgContext);
            SOAPEnvelope envelope = fac.getDefaultEnvelope();

            if (result != null) {
                envelope.getBody().addChild(result);
            }

            newmsgContext.setEnvelope(envelope);
            
        	// print the outgoing SOAP message as a debug message
        	logger.debug("Outgoing SOAP message: \n" + envelope);
        } catch (Exception e) {
        	logger.setResourceBundle(ResourceBundle.getBundle("org.eclipse.cosmos.dc.provisional.cmdbf.messages"));
        	if (e instanceof InvocationTargetException) {
        		Throwable targetException = ((InvocationTargetException)e).getTargetException();
        		if (targetException != null && targetException instanceof CMDBfServiceException) {
        			SOAPEnvelope envelope = getSOAPFactory(msgContext).getDefaultEnvelope();
        			OMElement fault = createFault(request.getOMFactory(), (CMDBfServiceException)targetException, logger);
                    if (fault != null) {
                        envelope.getBody().addChild(fault);
                    }
                    newmsgContext.setEnvelope(envelope);
                    logger.l7dlog(Level.ERROR, "OPERATION_ERROR", new Object[] {msgContext.getEnvelope().toString()}, targetException);
        		} else {
        			logger.l7dlog(Level.ERROR, "OPERATION_ERROR", new Object[] {msgContext.getEnvelope().toString()}, targetException);
        			throw AxisFault.makeFault(targetException);
        		}
        	} else {
        		logger.l7dlog(Level.ERROR, "OPERATION_ERROR", new Object[] {msgContext.getEnvelope().toString()}, e);
        		throw AxisFault.makeFault(e);
        	}
        }
    }
    
	protected SOAPFault createFault(OMFactory factory, CMDBfServiceException e, Logger logger) {
		
    	if (factory instanceof SOAP11Factory) {
    		SOAP11Factory soap11factory = (SOAP11Factory) factory;
    		return createSOAP11Fault(soap11factory, e, logger);
    	} else if (factory instanceof SOAP12Factory) {
    		SOAP12Factory soap12factory = (SOAP12Factory) factory;
    		return createSOAP12Fault(soap12factory, e, logger);
    	}
    	return null;
	}
	
	private SOAPFault createSOAP11Fault(SOAP11Factory soap11factory, CMDBfServiceException e, Logger logger) {
		// Get values from CMDBfServiceException
		String faultcode = null;
		QName subcodeqname = null;
		String reasonString = null;
		Node[] detailNodes = null;

		// code
		faultcode = getSOAP11Code(e.getCode());

		// subcode
		subcodeqname = getSubcodeQName(e.getSubCode());

		// reason
		reasonString = e.getMessage();
		
		// detail
		detailNodes = e.getDetails();
		
		// Create AXIOM SOAP objects
		SOAPFault soapfault = soap11factory.createSOAPFault();
		SOAPFaultDetail soapdetails = soap11factory.createSOAPFaultDetail(soapfault);
		SOAPFaultCode soapcode = soap11factory.createSOAPFaultCode(soapfault);
		SOAPFaultReason soapreason = soap11factory.createSOAPFaultReason(soapfault);
		
		// set code value
		soapcode.setText(faultcode);
		
		// set subcode
		// TODO: Subcode is not defined in the schema as of v1.0
		OMNamespace ns = soap11factory.createOMNamespace("http://cmdbf.org/schema/1-0-0/datamodel", "cmdbf");
		OMElement subcodeElement = soap11factory.createOMElement("Subcode", ns, soapdetails);
		soap11factory.createOMText(subcodeElement, subcodeqname);
		
		// set reason
		soapreason.setText(reasonString);
		
		// create details
		if (detailNodes != null) {
			setDetails(soapdetails, detailNodes, logger);
		}

        return soapfault;
	}

	private SOAPFault createSOAP12Fault(SOAP12Factory soap12factory, CMDBfServiceException e, Logger logger) {
		// Get values from CMDBfServiceException
		String faultcode = null;
		QName subcodeqname = null;
		String reasonString = null;
		Node[] detailNodes = null;

		// code
		faultcode = getSOAP12Code(e.getCode());

		// subcode (discarded)
		subcodeqname = getSubcodeQName(e.getSubCode());

		// reason
		reasonString = e.getMessage();
		
		// detail
		detailNodes = e.getDetails();

		// Create AXIOM SOAP objects
		SOAPFault soapfault = soap12factory.createSOAPFault();
		SOAPFaultDetail soapdetails = soap12factory.createSOAPFaultDetail(soapfault);
		SOAPFaultCode soapcode = soap12factory.createSOAPFaultCode(soapfault);
		SOAPFaultReason soapreason = soap12factory.createSOAPFaultReason(soapfault);
		SOAPFaultText soapfaultText = soap12factory.createSOAPFaultText(soapreason);
		SOAPFaultValue soapfaultcodevalue = soap12factory.createSOAPFaultValue(soapcode);

		// set reason
		soapfaultText.setText(reasonString);
		
		// set code
		soapfaultcodevalue.setText(faultcode);

		// create details
		if (detailNodes != null) {
			setDetails(soapdetails, detailNodes, logger);
		}

        // add the subcode last to make sure the subcode element is after the code value element
		SOAPFaultSubCode soapsubcode = soap12factory.createSOAPFaultSubCode(soapcode);
		SOAPFaultValue faultsubcodevalue = soap12factory.createSOAPFaultValue(soapsubcode);
		faultsubcodevalue.setText(subcodeqname);

		return soapfault;
	}
	
	private String getSOAP11Code(int code) {
		String faultcode = null;
		switch (code) {
		case CMDBfServiceException.SENDER:
			faultcode = SOAP11Constants.FAULT_CODE_SENDER;
			break;
		case CMDBfServiceException.RECIEVER:
			faultcode = SOAP11Constants.FAULT_CODE_RECEIVER;
			break;
		}
		return faultcode;
	}
	
	private String getSOAP12Code(int code) {
		String faultcode = null;
		switch (code) {
		case CMDBfServiceException.SENDER:
			faultcode = SOAP12Constants.FAULT_CODE_SENDER;
			break;
		case CMDBfServiceException.RECIEVER:
			faultcode = SOAP12Constants.FAULT_CODE_RECEIVER;
			break;
		}
		return faultcode;
	}
	
	private QName getSubcodeQName(int subcode) {
		QName subcodeqname = null;
		String dataModelNamespace = "http://cmdbf.org/schema/1-0-0/datamodel";
		switch (subcode) {
		case CMDBfServiceException.UNKNOWN_TEMPLATE_ID:
			subcodeqname = new QName(dataModelNamespace, "UnkownTemplateID", "cmdbf");
			break;
		case CMDBfServiceException.INVALID_PROPERTY_TYPE:
			subcodeqname = new QName(dataModelNamespace, "InvalidPropertyType", "cmdbf");
			break;
		case CMDBfServiceException.XPATH_ERROR:
			subcodeqname = new QName(dataModelNamespace, "XPathError", "cmdbf");
			break;
		case CMDBfServiceException.UNSUPPORTED_CONSTRAINT:
			subcodeqname = new QName(dataModelNamespace, "UnsupportedConstraint", "cmdbf");
			break;
		case CMDBfServiceException.UNSUPPORTED_SELECTOR:
			subcodeqname = new QName(dataModelNamespace, "UnsupportedSelector", "cmdbf");
			break;
		case CMDBfServiceException.QUERY_ERROR:
			subcodeqname = new QName(dataModelNamespace, "QueryError", "cmdbf");
			break;
		case CMDBfServiceException.INVALID_RECORD:
			subcodeqname = new QName(dataModelNamespace, "InvalidRecord", "cmdbf");
			break;
		case CMDBfServiceException.UNSUPPORTED_RECORD_TYPE:
			subcodeqname = new QName(dataModelNamespace, "UnsupportRecordType", "cmdbf");
			break;
		case CMDBfServiceException.INVALID_MDR:
			subcodeqname = new QName(dataModelNamespace, "InvalidMDR", "cmdbf");
			break;
		case CMDBfServiceException.REGISTRATION_ERROR:
			subcodeqname = new QName(dataModelNamespace, "RegisrationError", "cmdbf");
			break;
		case CMDBfServiceException.DEREGISTRATION_ERROR:
			subcodeqname = new QName(dataModelNamespace, "DeregistrationError", "cmdbf");
			break;
		}
		return subcodeqname;
	}
	
	private void setDetails(SOAPFaultDetail soapdetails, Node[] detailNodes, Logger logger) {
		for (int i=0; i<detailNodes.length; i++) {
			Node node = detailNodes[i];
			
			if (node instanceof Element) {
				Element elem = (Element) node;
				try {
					soapdetails.addChild(XMLUtils.toOM(elem));
				} catch (Exception e) {
					logger.l7dlog(Level.ERROR, "SET_DETAILS_ERROR", e);
				}
			} else {
				// TODO: turn node into stream, then use XMLUtils.toOM(InputStream) API
				// This is less efficient. 
				// TODO: TEST this code!
				try{
		            TransformerFactory tf = TransformerFactory.newInstance();
		            Transformer trans = tf.newTransformer();
		            PipedOutputStream os = new PipedOutputStream();
		            trans.transform(new DOMSource(node), new StreamResult(os));
		            InputStream is = new PipedInputStream(os);
		            soapdetails.addChild(XMLUtils.toOM(is));
				} catch (Exception e) {
					logger.l7dlog(Level.ERROR, "SET_DETAILS_ERROR", e);
				}
			}
		}
	}

}
