/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations.cmdbf;

import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfMessages;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility.NodeXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IPrefixMapping;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IXPathExpression;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.validation.artifacts.MappedNamespaceContext;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.w3c.dom.Node;

/**
 * The implementation of the XPath expression constraint
 * 
 * @author Ali Mehregani
 */
public class ItemXPathHandler extends AbstractSMLConstraintHandler
{
	/** 
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.impl.AbstractItemConstraintHandler#handle(org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes, org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	public INodes handle(INodes context, IConstraint constraint) throws CMDBfServiceException
	{	
		IXPathExpression xpathExpression = (IXPathExpression)constraint;
		
		// Make sure this is an XPath 1.0 expression
		if (!ICMDBfServicesConstants.XPATH_1_DIALECT.equals(xpathExpression.getDialect().toString()))
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(RepositoryMessages.errorUnsupportedXPathDialect));
		}
		
		try
		{	
			INodes result = retrieveContext(context);
			IGraphElement[] elements = result.getElements();
			
			// Build the name space context						
			MappedNamespaceContext namespaceContext = new MappedNamespaceContext();
			IPrefixMapping[] prefixMapping = xpathExpression.getPrefixMappings();
			for (int i = 0; i < prefixMapping.length; i++)
			{			
				namespaceContext.addEntry(prefixMapping[i].getPrefix(), prefixMapping[i].getNamespace().toString());
			}
			
			ISMLRepository repository = getRepository();
			List<IGraphElement> revisedList = new ArrayList<IGraphElement>();
			
			// For every document in the data set
			for (int i = 0; i < elements.length; i++)
			{			
				// Evaluate the XPath expression					
				synchronized (SMLValidatorUtil.xpath)
				{					
					SMLValidatorUtil.xpath.setNamespaceContext(namespaceContext);
					XPathExpression fieldXPath = SMLValidatorUtil.xpath.compile(xpathExpression.getExpression());	
					IItem item = (IItem)elements[i];
					Node rootNode = CMDBfUtil.retrieveDocumentNode(repository, item);
					if (rootNode == null)
					{
						continue;
					}
					String xpathResult = fieldXPath.evaluate(rootNode);
					Node xpathNodeResult = null;
					if (xpathResult == null || xpathResult.length() <= 0 || "false".equals(xpathResult))
					{
						// If this is not a boolean node, then it could be an empty element
						// that still exists.  Try to locate the node
						if (!"false".equals(xpathResult))
						{
							xpathNodeResult = (Node)fieldXPath.evaluate(rootNode, XPathConstants.NODE);
						}
						
						if (xpathNodeResult == null)
						{
							continue;
						}
					}
					else
					{
						// Find the corresponding node for evaluating the expression					
						if (!"true".equals(xpathResult))
						{
							xpathNodeResult = (Node)fieldXPath.evaluate(rootNode, XPathConstants.NODE);
						}
					}
					
					// Only one record is expected
					IRecord record = item.getRecords()[0];
					
					// Replace the value if this is the starting context
					if (context.isStartingContext())
					{
						record.setValue(new NodeXMLWritable(xpathNodeResult));
					}
					// Otherwise append the value
					else
					{
						if (record.getValue() instanceof NodeXMLWritable)
						{
							NodeXMLWritable nodeWritable = (NodeXMLWritable)record.getValue();
							nodeWritable.addNode(xpathNodeResult);
						}			
					}
					
					revisedList.add(item);
				}
			}
			
			result.setElements(revisedList.toArray(new IGraphElement[revisedList.size()]));
			return result;
		}	
		catch (XPathExpressionException e)
		{						
			throw new CMDBfServiceException
			(
				CMDBfServiceException.SENDER,
				CMDBfServiceException.XPATH_ERROR,
				CMDBfMessages.faultsQueryXPathError,
				new Node[]
				{ 
					// <cmdbf:expression> xs:string </cmdbf:expression>
					CMDBfInternalUtility.appendNode(
						CMDBfInternalUtility.createElement(
							ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE, 
							ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.EXPRESSION_ELEMENT),
						CMDBfInternalUtility.createTextNode(xpathExpression.getExpression())),
					
					// <cmdbf:xpathErrorCode> [xpath error code] </cmdbf:xpathErrorCode>
					CMDBfInternalUtility.appendNode(
						CMDBfInternalUtility.createElement(
							ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE, 
							ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.XPATH_ERROR_CODE_ELEMENT),
						CMDBfInternalUtility.createTextNode(e.getMessage())),									 
				},
				e
			);
		} 
		catch (RepositoryOperationException e)
		{			
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(e.getMessage()),
					e);
		}	
	}
}
