/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.common;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.TransformerException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecordSerializer;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Utility methods used by multiple components of
 * the query and registration service interfaces
 * 
 * 
 * @author David Whiteman
 */
public class CMDBfServicesUtil 
{

	private static String dataModelXsdFileName;

	private static IRecordSerializer recordSerializer;

	/**
	 * Create and return a URI from a String.  Return
	 * null if the URI is malformed.
	 * 
	 * @param uriString
	 * @return
	 */
	public static URI createURI(String uriString) {
		try	{
			uriString = uriString.replace(" ", "%20");
			return new URI(uriString);
		} catch (URISyntaxException e) {
		} 
		return null;
	}

	/**
	 * Return a string that forms a tag name for the namespace
	 * and elementName in namespace:elementName format.
	 * 
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String tagNameFor(String namespace, String elementName) {
		StringBuffer buffer = new StringBuffer("");
		if (namespace != null) {
			buffer.append(namespace).append(ICMDBfServicesConstants.COLON);
		}
		buffer.append(elementName);
		return buffer.toString();
	}

	/**
	 * Add <tt>indent</tt> number of tabs to the string managed by
	 * the StringWriter
	 * 
	 * @param stringWriter
	 * @param indent
	 */
	public static void addIndent(StringWriter stringWriter, int indent) {
		for (int i = 0; i < indent; i++) {
			stringWriter.append('\t');
		}
	}

	/**
	 * Return an XML begin tag for the elementName.  For example,
	 * with an argument of "foo", "<foo>" will be returned.
	 * 
	 * @param elementName
	 * @return
	 */
	public static String beginTagFor(String elementName) {
		return tagFor(false, null, elementName);
	}

	/**
	 * Return an XML end tag for the elementName.  For example,
	 * with an argument of "foo", "</foo>" will be returned.
	 * 
	 * @param elementName
	 * @return
	 */
	public static String endTagFor(String elementName) {
		return tagFor(true, null, elementName);
	}

	/**
	 * Utility method to return a begin or end tag for the namespace
	 * and elementName input.
	 * 
	 * @param end
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String tagFor(boolean end, String namespace, String elementName) {
		return tagFor(end, namespace, elementName, true);
	}
	
	/**
	 * Utility method to return a begin or end tag for the namespace
	 * and elementName input.
	 * 
	 * @param end Indicates whether this is an end tag
	 * @param namespace Optional namespace prefix
	 * @param elementName Name of element
	 * @param close Indicates whether the closing angle bracket should be added
	 * @return
	 */
	public static String tagFor(boolean end, String namespace, String elementName, boolean close) {
		StringBuffer tagBuffer = new StringBuffer();
		tagBuffer.append(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET);
		if (end) {
			tagBuffer.append(ICMDBfServicesConstants.FORWARD_SLASH);
		}
		tagBuffer.append(tagNameFor(namespace, elementName));
		if (close) {
			tagBuffer.append(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET);
		}
		return tagBuffer.toString();
	}

	/**
	 * Output all the IXMLWritable instances to the StringWriter in XML format.
	 * All API types that correspond to XML elements in the CMDBf spec are expected
	 * to implement IXMLWritable.
	 * 
	 * @param writer String writer on which the XML output is written
	 * @param indent Indent level to use for writing the writable
	 * @param writables API types that correspond to XML elements
	 */
	public static void outputListToXML(StringWriter writer, int indent, List<?> writables) {
		for (Iterator<?> iterator = writables.iterator(); iterator.hasNext();) {
			IXMLWritable output = (IXMLWritable) iterator.next();
			output.toXML(writer, indent);
		}
	}

	/**
	 * Write an XML element attribute to the string writer.  The format written will be
	 * attributeName="attributeValue"
	 * 
	 * @param writer
	 * @param attributeName
	 * @param attributeValue
	 */
	public static void writeAttribute(StringWriter writer, String attributeName, String attributeValue) {
		writer.write(attributeName+ICMDBfServicesConstants.EQUAL_SIGN+ICMDBfServicesConstants.DOUBLE_QUOTE);
		writer.write(attributeValue);
		writer.write(ICMDBfServicesConstants.DOUBLE_QUOTE);
	}

	/**
	 * Return an XML begin tag for the element name and the given attributes.  
	 * For example, with a name of "foo" and an Attributes containing a {bar, baz} pair,
	 * the following will be returned:
	 *  
	 * 	<foo bar="baz">
	 * 
	 * @param name
	 * @param attributes
	 * @return
	 */
	public static String beginTagFor(String name, Attributes attributes) {
		String startTag = tagFor(false, null, name, false);
		StringWriter output = new StringWriter();
		output.write(startTag);
		for (int i = 0; i < attributes.getLength(); i++) {
			output.write(ICMDBfServicesConstants.SINGLE_SPACE);
			writeAttribute(output, attributes.getQName(i), attributes.getValue(i));
		}
		output.write(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET);
		return output.toString();
	}

	/**
	 * Retrieve the schema document for the CMDBf data model.
	 * If the document is available locally, use that one.
	 * Otherwise load it from the CMDBf spec site.
	 * 
	 * @return
	 */
	public static InputStream getCMDBfDataModel() {
		File schemaFile = new File(getDataModelXsdFileName());
		InputStream inputStream = null;
		if (schemaFile.isAbsolute()) {
			try {
				inputStream = new FileInputStream(schemaFile);
			} catch (FileNotFoundException e) {
				// could not find, defer to web version
			}
		} else {
			// Do not use schemaFile.getPath(), as that uses platform path separators
			inputStream = CMDBfServicesUtil.class.getClassLoader().getResourceAsStream(getDataModelXsdFileName());
		}
		return inputStream;
	}

	/**
	 * Set the file name of the local file containing the data model
	 * schema.  This file name can be absolute, or relative to the class 
	 * loader for this class.
	 * 
	 * @param dataModelXsdFileName
	 */
	public static void setDataModelXsdFileName(String dataModelXsdFileName) {
		CMDBfServicesUtil.dataModelXsdFileName = dataModelXsdFileName;
	}

	/**
	 * Answer the file name of the local file containing the data model
	 * schema.  This file name can be absolute, or relative to the class 
	 * loader for this class.  If this field has not been set, the value
	 * defaults to a file path relative to the resources directory.
	 * 
	 * @return
	 */
	public static String getDataModelXsdFileName() {
		if (dataModelXsdFileName == null) {
			dataModelXsdFileName = ICMDBfServicesConstants.CMDBF_DATA_MODEL_XSD_FILE;
		}
		return dataModelXsdFileName;
	}

	/**
	 * Answer a new InputStream containing the XML representation of the model object
	 * 
	 * @param writable
	 * @return
	 */
	public static InputStream transformImpl(IXMLWritable writable) {
		StringWriter writer = new StringWriter();
		writable.toXML(writer, 0);
		return new ByteArrayInputStream(writer.getBuffer().toString().getBytes());
	}

	/**
	 * Transform the XML content into an object graph.  The top level object must implement
	 * IRootElement, and the SAX parser handler must implement ITransformerHandler.  The parser
	 * performs validation on the XML using the CMDBf data model schema.
	 * 
	 * @param inputSource
	 * @param handler
	 * @return
	 * @throws TransformerException
	 */
	public static IRootElement transformImpl(InputSource inputSource, ITransformerHandler handler) 
			throws TransformerException {
		return transformImpl(inputSource, handler, null);
	}
	
	/**
	 * @param inputSource
	 * @param handler
	 * @param transformOptions a Map of options including whether to validate and also whether to
	 * pass in a local data model stream
	 * @return
	 * @throws TransformerException
	 */
	public static IRootElement transformImpl(InputSource inputSource, ITransformerHandler handler, Map<String, Object> transformOptions)
				throws TransformerException {
		
		boolean validating = true;
		InputStream dataModelStream = null;
		
		if (transformOptions != null) {
			Object validateOption = transformOptions.get(ICMDBfServicesConstants.SERVICES_TRANSFORM_VALIDATION_OPTION);
			if (validateOption != null) {
				validating = ((Boolean) validateOption).booleanValue();
			}
			Object dataModelOption = transformOptions.get(ICMDBfServicesConstants.SERVICES_TRANSFORM_DATAMODEL_OPTION);
			if (dataModelOption != null) {
				dataModelStream = (InputStream) dataModelOption;
			}
		}
		if (validating && (dataModelStream == null)) {
			dataModelStream = getCMDBfDataModel();
			// If we can't locate the data model, 
			validating = (dataModelStream != null);
		}

		try {
			SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
			saxParserFactory.setNamespaceAware(true);
		    saxParserFactory.setValidating(validating);

			saxParserFactory.setFeature(ICMDBfServicesConstants.SAX_PARSER_FEATURE_VALIDATION, validating);
//			saxParserFactory.setFeature("http://apache.org/xml/features/validation/schema", true);
//			saxParserFactory.setFeature("http://apache.org/xml/features/validation/schema-full-checking", true);
			saxParserFactory.setFeature(ICMDBfServicesConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(ICMDBfServicesConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
			
			SAXParser parser = saxParserFactory.newSAXParser();
			if (validating) {
				parser.setProperty(ICMDBfServicesConstants.SAX_PARSER_PROPERTY_SCHEMA_LANGUAGE, ICMDBfServicesConstants.SCHEMA_URI); 
				parser.setProperty(ICMDBfServicesConstants.SAX_PARSER_PROPERTY_SCHEMA_SOURCE, dataModelStream);					
			}
			if (handler instanceof LexicalHandler) {
				XMLReader reader = parser.getXMLReader();
				reader.setProperty("http://xml.org/sax/properties/lexical-handler", (LexicalHandler) handler);
			}
			parser.parse(inputSource, (DefaultHandler) handler);
		} catch (SAXException e) {
			throw new TransformerException(e);
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} 
		return handler.getResult();
	}

	/**
	 * Answer the record serializer used to transform IRecord
	 * content to a user object.
	 * 
	 * @see #setRecordSerializer(IRecordSerializer)
	 * @return
	 */
	public static IRecordSerializer getRecordSerializer() {
		return recordSerializer;
	}

	/**
	 * Set the serializer used to convert a String representation of
	 * IRecord content to a user object.  That serializer must also implement
	 * IXMLWritable to conduct the reverse transformation of writing the user
	 * object to XML.
	 * 
	 * @param recordSerializer
	 */
	public static void setRecordSerializer(IRecordSerializer recordSerializer) {
		CMDBfServicesUtil.recordSerializer = recordSerializer;
	}
	
	
	/**
	 * Returns a string friendly version of the uri passed in
	 * 
	 * @param uri The uri
	 * @return A string friendly version of uri
	 */
	public static String toString(URI uri)
	{
		try
		{
			if (uri == null)
			{
				return null;
			}
			return URLDecoder.decode(uri.toString(), "UTF-8");
		} 
		catch (UnsupportedEncodingException e)
		{
			return uri.toString();
		}
	}
}
