/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.registration;

import java.io.InputStream;
import java.io.StringReader;
import java.util.ResourceBundle;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.context.ServiceContext;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.INotificationHandlerFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.service.ICMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.service.impl.CMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.transform.DeregistrationOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.service.ICMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.service.impl.CMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.RegistrationOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;

/**
 * Abstract implementation of a CMDBf registration service. 
 * Implementations of a CMDBf registration web service will subclass from this class. 
 *
 */
public abstract class AbstractRegistrationService {

	/**
	 * The registration operation
	 */
	private ICMDBfRegistrationOperation registrationOperation;
	
	/**
	 * The deregistration operation
	 */
	private ICMDBfDeregistrationOperation deregistrationOperation;

	protected Logger logger = null;
	
	/**
	 * Life cycle method of the web service.  This method is called by Axis2 framework. 
	 * @param serviceContext
	 */
	public void init(ServiceContext serviceContext) {
    	logger = Logger.getLogger(getClass());
    	logger.setResourceBundle(ResourceBundle.getBundle("org.eclipse.cosmos.dc.provisional.cmdbf.messages"));
    }
	
	/**
	 * Default constructor
	 */
	public AbstractRegistrationService()
	{
		INotificationHandlerFactory factory = getNotificationHandlerFactory();
		registrationOperation = new CMDBfRegistrationOperation(factory, null);
		deregistrationOperation = new CMDBfDeregistrationOperation(factory, null);
	}

	/**
	 * CMDBf registration operation
	 * 
	 * @param registerRequest registration request (SOAP body of incoming SOAP message)
	 * @return registration response
	 * @throws CMDBfServiceException error occurred during registration
	 */
	public OMElement Register(OMElement registerRequest) throws CMDBfServiceException {
		
		OMElement documentElement = null;
		IRegisterResponse result = null;
		
		result = registrationOperation.execute(new StringReader(registerRequest.toString())); 

		InputStream output = RegistrationOutputTransformer.transform(result);
		XMLStreamReader parser = null;
		try {
			parser = XMLInputFactory.newInstance().createXMLStreamReader(output);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
			//get the root element
			documentElement =  builder.getDocumentElement();
		} catch (XMLStreamException e) {
			logger.l7dlog(Level.ERROR, "XMLStreamReader_ERROR", e);
			throw new CMDBfServiceException(CMDBfServiceException.SENDER, 
					CMDBfServiceException.REGISTRATION_ERROR,
					e.getMessage());
		}
		return documentElement;
	}

	/**
	 * CMDBf deregistration operation
	 * @param deregisterRequest deregistration request
	 * @return deregistration response
	 * @throws CMDBfServiceException error occurred during deregistration
	 */
	public OMElement Deregister(OMElement deregisterRequest) throws CMDBfServiceException {
		
		OMElement documentElement = null;
		IDeregisterResponse result = null;
		
		result = deregistrationOperation.execute(new StringReader(deregisterRequest.toString())); 

		InputStream output = DeregistrationOutputTransformer.transform(result);
		XMLStreamReader parser = null;
		try {
			parser = XMLInputFactory.newInstance().createXMLStreamReader(output);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
			//get the root element
			documentElement =  builder.getDocumentElement();
		} catch (XMLStreamException e) {
			logger.l7dlog(Level.ERROR, "XMLStreamReader_ERROR", e);
			throw new CMDBfServiceException(CMDBfServiceException.SENDER, 
					CMDBfServiceException.DEREGISTRATION_ERROR,
					e.getMessage());
		}
		return documentElement;
	}

	/**
	 * This method is expected to be implemented by subclasses.
	 * The method returns a notification handler factory that is needed
	 * for the registration/deregistration operations
	 * 
	 * @return The notification handler factory
	 */
	public abstract INotificationHandlerFactory getNotificationHandlerFactory();

	/**
	 * @return the registrationOperation
	 */
	public ICMDBfRegistrationOperation getRegistrationOperation() {
		return registrationOperation;
	}

	/**
	 * @return the deregistrationOperation
	 */
	public ICMDBfDeregistrationOperation getDeregistrationOperation() {
		return deregistrationOperation;
	}

}
