/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.query;

import java.io.InputStream;
import java.io.StringReader;
import java.util.ResourceBundle;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.context.ServiceContext;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IQueryHandlerFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.impl.CMDBfQueryOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.QueryOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.w3c.dom.Node;

/**
 * Abstract implementation of a CMDBf query service. 
 * Implementations of a CMDBf query web service will subclass from this class. 
 *
 */
public abstract class AbstractQueryService {
	protected CMDBfQueryOperation queryOp = new CMDBfQueryOperation(getQueryHandlerFactory(), null);
	
	protected Logger logger = null;
	/**
	 * This method is expected to be implemented by subclasses.
	 * The method returns a query handler factory that is needed
	 * for the graph query operations
	 * @return
	 */
	abstract protected IQueryHandlerFactory getQueryHandlerFactory();
	
	/**
	 * Life cycle method of the web service.  This method is called by Axis2 framework. 
	 * 
	 * @param serviceContext
	 */
	public void init(ServiceContext serviceContext) {
    	logger = Logger.getLogger(getClass());
    	logger.setResourceBundle(ResourceBundle.getBundle("org.eclipse.cosmos.dc.provisional.cmdbf.messages"));
    }
    
	/**
	 * 
	 * @return the query operation
	 */
	protected CMDBfQueryOperation getQueryOperation() {
		return queryOp;
	}

	/**
	 * Default implementation of the graph query operation. 
	 * 
	 * @param query query request (SOAP body of incoming SOAP message)
	 * @return query response 
	 * @throws CMDBfServiceException error occurred during query operation
	 */
	public org.apache.axiom.om.OMElement GraphQuery(
			org.apache.axiom.om.OMElement query) throws CMDBfServiceException {
		
		OMElement documentElement = null;
		IQueryResult result = null;

		result = queryOp.execute(new StringReader(query.toString()));
		
		InputStream output = QueryOutputTransformer.transform(result);
		XMLStreamReader parser = null;
		try {
			parser = XMLInputFactory.newInstance().createXMLStreamReader(output);
		} catch (XMLStreamException e) {
			logger.l7dlog(Level.ERROR, "XMLStreamReader_ERROR", e);
			throw new CMDBfServiceException(CMDBfServiceException.RECIEVER, 
					CMDBfServiceException.QUERY_ERROR,
					e.getMessage(),
					(Node)null,
					e);
		}
		//create the builder
		StAXOMBuilder builder = new StAXOMBuilder(parser);
		//get the root element
		documentElement =  builder.getDocumentElement();
		return documentElement;
	}
	
}