/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.resource;

import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.repository.core.MetadataProcessor;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;

/**
 * The file system implementation of the SML document meta-data
 * 
 * @author Ali Mehregani
 */
public class SMLFileMetadata implements ISMLMetadata
{
	/**
	 * The id of the document
	 */
	private String id;
	
	/**
	 * The document type
	 */
	private int documentType;

	/**
	 * Root element name
	 */
	private String rootElementName;

	/**
	 * Meta-data Processor
	 */
	private MetadataProcessor metadataProcessor;
	
	/**
	 * The aliases
	 */
	private String[] aliases;
	
	/**
	 * The rule bindings
	 */
	private Map<String, List<String>> rules;
	
	public SMLFileMetadata()
	{
		documentType = -1;
	}
		
	/**
	 * A convenience constructor used to set the fields of the meta-data
	 * 
	 * @param id The id of the document
	 * @param type The type of the document
	 * @param elementName The element name
	 * @param aliases Aliases of the document
	 * @param rules Rules of the document
	 */
	public SMLFileMetadata(String id, int type, String 
						elementName, String[] aliases, Map<String, List<String>> rules)
	{
		super();
		this.id = id;
		this.documentType = type;
		this.rootElementName = elementName;
		this.aliases = aliases;
		this.rules = rules;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata#getAliases()
	 */
	public String[] getAliases()
	{
		return aliases == null ? metadataProcessor == null ? null : metadataProcessor.retrieveAliases(id) : aliases;
	}
	
	public void setAliases(String[] aliases)
	{
		this.aliases = aliases;
	}


	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata#getBoundRules()
	 */
	public Map<String, List<String>> getBoundRules()
	{
		return rules == null ? metadataProcessor == null ? null : metadataProcessor.retrieveBoundRules(id) : rules;
	}

	public void setBoundRules(Map<String, List<String>> rules)
	{
		this.rules = rules;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata#getDocumentType()
	 */
	public int getDocumentType()
	{
		return documentType;
	}
	
	public void setDocumentType(int documentType)
	{
		this.documentType = documentType;		
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata#getId()
	 */
	public String getId()
	{
		return id;
	}

	
	public void setId(String id)
	{
		this.id = id;	
	}
		
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata#getRootElementName()
	 */
	public String getRootElementName()
	{
		return rootElementName;
	}

	public void setRootElementName(String rootElementName)
	{
		this.rootElementName = rootElementName;
	}

	public void setRepositoryMetadata(MetadataProcessor metadataProcessor)
	{
		this.metadataProcessor = metadataProcessor;
	}
}
