/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.common.validation.plugin;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.cosmos.me.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.sdd.common.validation.XML_DAS;
import org.eclipse.cosmos.me.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

/**
 * 
 * @author Eric S. Rose (esrose@us.ibm.com)
 *
 */
public class XML_DASImpl implements XML_DAS {
	// DAS contains a list of ValidationRules
	private Collection<ValidationRule> validationRuleList = new Vector<ValidationRule>();
	
	public XML_DASImpl() {
	}

	public void addValidation(ValidationRule validationRule) {
		// Add the given validation rule to the list
		validationRuleList.add(validationRule);
	}
	
	public void removeValidation(ValidationRule validationRule) {
		// Remove the given validation rule from the list
		validationRuleList.remove(validationRule);
	}
	
	public Collection<ValidationRule> getValidationRules() {
		// Return the list of ValidationRules
		return validationRuleList;
	}

	public Document loadDocument(InputStream inputStream) throws IOException, XMLValidationException {
		// Build a Document from the InputStream
		Document xmlDocument = null;
		
		try {
			// Create a DocumentBuilder and parse the InputStream into a Document
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			xmlDocument = builder.parse(inputStream);		
		
		} catch (SAXException e) {
			// Thrown if the input XML is not well-formed
			
			// Throw an XMLValidationException containing the SAXException
			XMLValidationError validationError = new XMLValidationError("", null, e);
			throw new XMLValidationException(validationError);
		
		} catch (ParserConfigurationException e) {
			// Thrown if Factory can't handle requested config (incompatibility with Java implementation)

			// Log the exception
			e.printStackTrace();
			
		} finally {
			// Close the input stream
			inputStream.close();
		}
		
		removeWhitespaceNodes(xmlDocument.getDocumentElement());
		return xmlDocument;
	}

	public void saveDocument(Document document, OutputStream outputStream, String encoding, boolean performValidation)
			throws XMLValidationException, IOException {
		if (performValidation) {
			// Perform the validation if instructed to do so
			Collection<XMLValidationError> validationErrors = validate(document);
			
			if (validationErrors.size() > 0) {
				// If validation errors occurred, then a XMLValidationException containing the errors
				throw new XMLValidationException(validationErrors);
			}
		}

		// If we get here, no exception has been thrown and it's okay to save the Document
		writeDocumentToOutputStream(document, outputStream, encoding);
	}

	public Collection<XMLValidationError> validate(Document document) {
		// Build an OutputStream and dump the XML to it
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		try {
			writeDocumentToOutputStream(document, outputStream, null);
		} catch (IOException e) {
			// This only happens if we run out of memory (give up?)
			e.printStackTrace();
		}
		
		Collection<XMLValidationError> validationErrors = new Vector<XMLValidationError>();

		// Run each of the validation rules
		for (ValidationRule validationRule : validationRuleList) {
			// Build an InputStream from the XML data and validate it
			ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
			validationErrors.addAll(validationRule.validate(inputStream));
		}

		// Return the list of validation errors (which may be empty)
		return validationErrors;
	}
	
	/*
	 * Write the given Document to the given OutputStream using the given encoding
	 */
	private void writeDocumentToOutputStream(Document document, OutputStream outputStream, String encoding)
			throws IOException {
		// Use a default encoding if none was specified
		if (encoding == null) {
			encoding = "UTF-8";
		}
		
		// Build a Transformer with the appropriate properties to transform the Document to XML output
		Transformer tr = null;
		try {
			tr = TransformerFactory.newInstance().newTransformer();
			tr.setOutputProperty(OutputKeys.INDENT, "yes");
			tr.setOutputProperty(OutputKeys.METHOD, "xml");
			tr.setOutputProperty(OutputKeys.ENCODING, encoding);
			
		} catch (TransformerConfigurationException e) {
			// There's an incompatibility with this Java implementation 
			e.printStackTrace();
			
		} catch (TransformerFactoryConfigurationError e) {
			// There's an incompatibility with this Java implementation
			e.printStackTrace();
			
		} catch (IllegalArgumentException e) {
			// There's an incompatibility with this Java implementation
			e.printStackTrace();			
		}

		// Write the Document to the OutputStream
		try {
			tr.transform(new DOMSource(document), new StreamResult(outputStream));
		} catch (TransformerException e) {
			// There's an incompatibility with this Java implementation
			e.printStackTrace();
		}
	}
	
	/*
	 * Remove all whitespace-only Text nodes from a Document Element
	 */
	private void removeWhitespaceNodes(Element e) {
		NodeList children = e.getChildNodes();
		
		for (int i = children.getLength() - 1; i >= 0; i--) {
			Node child = children.item(i);
			
			if (child instanceof Text && ((Text)child).getData().trim().length() == 0) {
				e.removeChild(child);
			}
			else if (child instanceof Element) {
				removeWhitespaceNodes((Element)child);
			}
		}
	}
}