/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.internal.validation.common.XMLInternalUtility;
import org.w3c.dom.Node;

/**
 * Model object to track instances found in validating SML resources.
 * It is used by the SchemaValidator in attempt to preserve the original
 * source to improve the accuracy of line number information in
 * validation error messages.
 * 
 * @author David Whiteman
 * @author John Arwe
 */
public class ElementModel extends ElementLocation {

	private String source;								//	Contents of the document, potentially modified to add schemaLocations 
	private String elementName;						//	smlif:data etc
	private int endingLine;
	private int endingColumn;
	private String additionalSchemaLocation;	//	Validator-added schemalocation default(s), e.g. for SML schema if imported without a schemaLocation
	
	/**
	 * A computed value containing the list of definition aliases
	 * required to validate this instance
	 * As a first step, find all schemaBindings where any of the element's aliases
	 * is bound to a schema.
	 * Second, add all defaultSchema bindings schemas as defined in the defaultSchemaBinding element
	 * Last, for all schemas found in step one, get all imported or included schemas
	 * Note: if the schemaBindings is missing completely, step 1 and 2 are replaced by namespace
	 * match ( find all schemas with a targetNamespace = ns where ns is used by this instance
	 * 
	 */
	private List<String> calculatedValidationSchemas;
	
	/**
	 * the aliases of the definition or instance document
	 */
	private List<String> aliases;
	
	/**
	 * The document base URI used (if necessary) to transform relative references into 
	 * absolute URIs.  The value may be derived from one or more of the following, in
	 * priority order (i.e. the first one of them with an absolute URI, if any, stops the process):
	 * 1.	xml:base value
	 * 2.	smlif:baseURI value from the model document's /smlif:docinfo/smlif:baseURI
	 * 3.	smlif:baseURI value from the SML-IF document's /smlif:model/smlif:identity/smlif:baseURI
	 * If none of them have been encountered yet (e.g. all URIs in the document are already absolute),
	 * its value is null.
	 * The value may also be a relative reference itself.  This is permissible as long as the value
	 * is never used to transform a relative reference to an absolute URI (if that occurs, the initiator
	 * of the transformation process must catch and report the error when it actually occurs).
	 */
	private URI documentBaseURI;
	

	public ElementModel(String name, String filePath, int lineNumber, int columnNumber) {
		super(filePath, lineNumber, columnNumber);
		this.elementName = name;
		
		aliases = new ArrayList<String>();		
		documentBaseURI = null;
	}

	public ElementModel(Node node, String filePath, int lineNumber, int columnNumber)
	{
		this (node.getNodeName(), filePath, lineNumber, columnNumber);
		
		StringBuffer sb = new StringBuffer();		
		XMLInternalUtility.serializeNode(sb, node);
		setSource(sb.toString());
	}

	public int getEndingLine() {
		return endingLine;
	}

	public void setEndingLine(int endingLine) {
		this.endingLine = endingLine;
	}

	public String getSource() {
		return source;
	}

	public void setSource(String contents) {
		this.source = contents;
	}

	public String getElementName() {
		return elementName;
	}

	public int getEndingColumn() {
		return endingColumn;
	}

	public void setEndingColumn(int columnNumber) {
		this.endingColumn = columnNumber;
	}

	public int numberOfLines() {
		if (getLineNumber() > getEndingLine()) {
			return 0;
		}
		return getEndingLine() - getLineNumber() + 1;
	}

	public String toString() {
		return getClass().getSimpleName() + " <"+elementName+"> line="+getLineNumber()+", col="+getColumnNumber();
	}

	public void setAdditionalSchemaLocation(String string) {
		additionalSchemaLocation = string;
	}

	public String getAdditionalSchemaLocation() {
		return additionalSchemaLocation;
	}
	
	public boolean equals(Object o)
	{
		if (!(o instanceof ElementModel))
			return false;
		
		if (this == o)
			return true;
		
		ElementModel elementModel = (ElementModel)o;		
		return 	(elementName.equals(elementModel.getElementName())) &&
				(isEquals(aliases, elementModel.aliases)) &&
				(new Path (getFilePath()).equals(new Path(elementModel.getFilePath()))) &&
				(source == null ? elementModel.getSource() == null : source.equals(elementModel.getSource()));
	}
	
	/**
	 * Returns true iff the length of the alias list is the same
	 * and they contain the same entries
	 * 
	 * @param aliases1 The first alias list
	 * @param aliases2 The second alias list
	 * @return true iff aliases1 is the same as aliases2
	 */
	private boolean isEquals(List<String> aliases1, List<String> aliases2)
	{
		int size = 0;
		if ((aliases1 == null && aliases2 != null) || 
			(aliases2 == null && aliases1 != null) ||
			((size = aliases1.size()) != aliases2.size()))
		{
			return false;
		}
		
		for (int i = 0; i < size; i++)
		{
			if (!aliases1.contains(aliases2.get(i)))
			{
				return false;
			}
		}
		
		return true;
	}

	/**
	 * @return the aliases
	 */
	public String[] getAliases() {
		return aliases.toArray(new String[aliases.size()]);
	}

	/**
	 * @param aliases the aliases to set
	 */
	public void setAliases(List<String> aliases) {
		this.aliases.addAll(aliases);
	}

	/**
	 * @return the calculatedValidationSchemas
	 */
	public List<String> getCalculatedValidationSchemas() {
		return calculatedValidationSchemas;
	}

	/**
	 * @param calculatedValidationSchemas the calculatedValidationSchemas to set
	 */
	public void setCalculatedValidationSchemas(List<String> calculatedValidationSchemas) {
		this.calculatedValidationSchemas = calculatedValidationSchemas;
	}

	
	/**
	 * Add an alias to this element model
	 * 
	 * @param alias The alias to be added 
	 */
	public void addAlias(String alias)
	{
		aliases.add(alias);
	}
	
	/**
	 * Get the document base URI
	 * 
	 * @return the document base URI
	 */
	public URI getDocumentBaseURI()
	{
		return documentBaseURI;
	}
	
	/**
	 * Set the document base URI
	 * 
	 * @param baseURI The URI to be used to transform relative references within the 
	 * document into absolute URIs, in the absence of other over-riding base URIs lower in the document tree.
	 */
	public void setDocumentBaseURI(URI baseURI)
	{
		documentBaseURI = baseURI;
	}
	
}
