/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

/**
 * Represents a validation set
 * 
 * @author Ali Mehregani
 */
public class ValidationSet
{
	/**
	 * The definition documents
	 */
	private List<ElementSchemaModel> definitions;
	
	/**
	 * The definition documents indexed by namespace
	 */
	private Map<String, ElementSchemaModel> definitionsByNamespace;
	
	/**
	 * The instance documents
	 */
	private List<ElementModel> instances;
	

	/**
	 * Constructor
	 */
	public ValidationSet()
	{
		this.definitions = new ArrayList<ElementSchemaModel>();
		this.instances = new ArrayList<ElementModel>();
		this.definitionsByNamespace = new Hashtable<String, ElementSchemaModel>();
	}
	
	
	/**
	 * Add a definition document
	 * 
	 * @param document The definition document to add
	 */
	public void addDefinitionDocument(ElementSchemaModel document)
	{
		this.definitionsByNamespace.put(document.getTargetNamespace(), document);
		this.definitions.add(document);
	}
	
	
	/**
	 * Add an instance document
	 * 
	 * @param document The instance document to add
	 */
	public void addInstanceDocument(ElementModel document)
	{
		this.instances.add(document);
	}


	/**
	 * Retrieves and returns the definition document included
	 * in this validation set.
	 * 
	 * @return The definitions of this validation set
	 */
	public ElementSchemaModel[] getDefinitions()
	{
		return definitions.toArray(new ElementSchemaModel[definitions.size()]);
	}


	/**
	 * Retrieves a definition document based on a namespace
	 * 
	 * @param namespace The namespace used for lookup
	 * @return The definition document associated with namespace
	 */
	public ElementSchemaModel getDefinitionByNamespace(String namespace)
	{
		return definitionsByNamespace.get(namespace);
	}


	/**
	 * Get all instances
	 * 
	 * @return all instances
	 */
	public ElementModel[] getInstances()
	{
		return instances.toArray(new ElementModel[instances.size()]);
	}


	/**
	 * Add multiple definition documents
	 * 
	 * @param definitions The definitions to be added
	 */
	public void addDefinitionDocuments(ElementSchemaModel[] definitions)
	{
		for (int i = 0; i < definitions.length; i++)
		{
			addDefinitionDocument(definitions[i]);
		}
	}


	/**
	 * Add multiple instance documents
	 * 
	 * @param instances The instances to be added
	 */
	public void addInstanceDocuments(ElementModel[] instances)
	{
		for (int i = 0; i < instances.length; i++)
		{
			addInstanceDocument(instances[i]);
		}
	}
	
	
	/**
	 * Returns true only if this validation set does
	 * not contain any documents
	 * 
	 * @return true if this validation set is empty;
	 * false otherwise
	 */
	public boolean isEmpty()
	{
		return (definitions.size() + instances.size()) == 0;
	}
}
