/**********************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.AbstractDeclaration;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.GroupDeclaration;
import org.eclipse.cosmos.rm.internal.validation.artifacts.GroupDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeDeclaration;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

//TODO: Ali M.: Will likely be removed
/**
 * This builder is used to construct a data structure that keeps track
 * of all type declarations.
 *
 * @author Ali Mehregani
 * @author John Arwe
 */
public class TypeDeclarationBuilder extends AbstractDeclarationBuilder<TypeDeclarationCollection>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".TypeDeclarationBuilder";

	/**
	 * The data structure used to keep track all type declarations
	 */
	private TypeDeclarationCollection datastructure;


	public TypeDeclarationBuilder()
	{
		datastructure = new TypeDeclarationCollection();
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	@Override
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{
		super.startElement(uri, localName, name, attributes);

		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			return;
		}

		ElementPSVI elementPsvi = getPsvi().getElementPSVI();
		if (elementPsvi == null) return;

		XSTypeDefinition typeDefinition = elementPsvi.getTypeDefinition();
		if (typeDefinition == null) return;

		if (typeDefinition.getTypeCategory() != XSTypeDefinition.COMPLEX_TYPE) {
			return;
		}

		setDeclaration(setCommonFields(typeDefinition, new TypeDeclaration()));

		if (getDeclaration() == null) return;

		// handle restriction case

		if (true) return;

		// <xs:complexType
		if (ISMLConstants.COMPLEXTYPE_ELEMENT.equals(localName))
		{
			setDeclaration(setCommonFields(attributes, new TypeDeclaration()));
		}
		else if (getDeclaration() != null)
		{
			TypeDeclaration typeDeclaration = (TypeDeclaration)getDeclaration();

			// <xs:restriction
			if (ISMLConstants.RESTRICTION_ELEMENT.equals(localName))
			{
				// Determine if this type declaration is derived from another type declaration
				QName baseRestriction = SMLValidatorUtil.toQName(getPrefixMap(), getTargetNamespace(), retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.BASE_ATTRIBUTE));
				if (baseRestriction != null)
				{
					TypeDeclaration baseRestrictionDeclaration = datastructure.get(baseRestriction);
					if (baseRestrictionDeclaration != null)
					{
						typeDeclaration.setBaseRestriction(baseRestrictionDeclaration);
					}
				}
			}
		}
	}


	private AbstractDeclaration setCommonFields(XSTypeDefinition typeDefinition, TypeDeclaration declaration) {
		String typeName = typeDefinition.getName();
		String typeUri = typeDefinition.getNamespace();

		if (typeName != null && typeUri != null)
		{
			declaration.setQName(new QName(typeUri, typeName));
			declaration.setLocation(new ElementLocation(null, getLocator().getLineNumber(), getLocator().getColumnNumber()));
			return declaration;
		}
		return null;
	}


	@Override
	public void endElement(String uri, String localName, String name) throws SAXException
	{
		super.endElement(uri, localName, name);

		if (ISMLConstants.SCHEMA_URI.equals(uri) || getDeclaration() == null)
		{
			return;
		}

		datastructure.add((TypeDeclaration)getDeclaration());
		setDeclaration(null);
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public TypeDeclarationCollection getDataStructure()
	{
		return datastructure;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#isStructureValid()
	 */
	@Override
	public boolean isStructureValid()
	{
		// For each type declaration
		for (Iterator<TypeDeclaration> iterator = datastructure.iterator(); iterator.hasNext();)
		{
			TypeDeclaration typeDeclaration = iterator.next();

			// Recursively check the consistency of restriction hierarchy
			if (!isConsistent(typeDeclaration))
			{
				return false;
			}
		}
		return true;
	}


	private boolean isConsistent(TypeDeclaration typeDeclaration)
	{
		// Resolve the element declarations associated with this type declaration
		// and make sure that the declarations are consistent locally
		Map<QName, List<ConstraintNode>> elementDeclarations = retrieveDuplicateDeclarations(typeDeclaration);
		if (!isConsistent(elementDeclarations))
		{
			setErrorMessage(ValidationMessageFactory.createErrorMessage(
					typeDeclaration.getLocation().getLineNumber(),
					NLS.bind(SMLValidationMessages.targetInconsistentDeclarations, typeDeclaration.getQName().toString())));
			return false;
		}


		// Check to ensure that the element declarations of this type is consistent
		// with its base restriction type
		TypeDeclaration baseRestriction = typeDeclaration.getBaseRestriction();
		if (baseRestriction == null)
			return true;

		// For every local element declaration of typeDeclaration
		for (Iterator<ConstraintNode> localElementDeclarations = typeDeclaration.getElementDeclarations().iterator(); localElementDeclarations.hasNext();)
		{
			// Check the consistency of the local element declaration with it's corresponding one
			// owned by the parent of typeDeclaration
			ConstraintNode localElementDeclaration = localElementDeclarations.next();
			ConstraintNode parentLocalElementDeclaration = baseRestriction.getElementDeclarations().get(new QName(localElementDeclaration.getUri(), localElementDeclaration.getName()));			
			if (parentLocalElementDeclaration != null && !isEqual(localElementDeclaration, parentLocalElementDeclaration))
			{
				setErrorMessage(ValidationMessageFactory.createErrorMessage(typeDeclaration.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.targetBadDerivation, typeDeclaration.getQName().toString())));
				return false;
			}
		}

		return true;
	}


	/**
	 * Given a type declaration, this helper method retrieves and
	 * returns all element declarations that are encountered more
	 * than once
	 *
	 * @param typeDeclaration The type declaration
	 * @return Element declarations associated with this type declaration.
	 */
	private Map<QName, List<ConstraintNode>> retrieveDuplicateDeclarations(TypeDeclaration typeDeclaration)
	{
		Map<QName, List<ConstraintNode>> duplicates = new Hashtable<QName, List<ConstraintNode>>();
		List<ElementDeclarationCollection> allDeclarations = new ArrayList<ElementDeclarationCollection>();
		ElementDeclarationCollection localCollection = typeDeclaration.getElementDeclarations();
		if (localCollection != null)
		{
			allDeclarations.add(localCollection);
		}

		// For every group
		GroupDeclarationCollection groupCollection = typeDeclaration.getGroupDeclarations();
		for (Iterator<GroupDeclaration> groups = groupCollection.iterator(); groups.hasNext();)
		{
			// Resolve the group if it's a reference
			GroupDeclaration group = groups.next();
			if (group.getRef() != null)
			{
				group = DataBuilderUtil.retrieveGroup(group.getRef());
			}
			if (group == null)
			{
				continue;
			}

			// For each element declaration of the group
			ElementDeclarationCollection groupElementCollection = group.getElementDeclarations();
			for (Iterator<ConstraintNode> iterator = groupElementCollection.iterator(); iterator.hasNext();)
			{
				ConstraintNode groupElementDeclaration = iterator.next();
				addDuplicates (duplicates, allDeclarations, groupElementDeclaration);
			}
			allDeclarations.add(groupElementCollection);
		}
		return duplicates;
	}


	private void addDuplicates(Map<QName, List<ConstraintNode>> duplicates, List<ElementDeclarationCollection> allDeclarations, ConstraintNode element)
	{
		// For each element declaration collection
		QName qName = element.getName() == null ? element.getRef() : new QName(element.getUri(), element.getName());
		boolean elementAdded = false;
		for (Iterator<ElementDeclarationCollection> declarations = allDeclarations.iterator(); declarations.hasNext();)
		{
			ElementDeclarationCollection elementDeclarationCollection = (ElementDeclarationCollection) declarations.next();
			ConstraintNode currentElement = elementDeclarationCollection.get(qName);
			if (currentElement != null)
			{
				List<ConstraintNode> list = duplicates.get(qName);
				if (list == null)
				{
					list = new ArrayList<ConstraintNode>();
					duplicates.put(qName, list);
				}

				list.add(currentElement);
				if (!elementAdded)
				{
					elementAdded = true;
					list.add(element);
				}
			}
		}
	}


	private boolean isConsistent(Map<QName, List<ConstraintNode>> elementDeclarations)
	{
		for (Iterator<QName> qNames = elementDeclarations.keySet().iterator(); qNames.hasNext();)
		{
			QName qName = qNames.next();
			List<ConstraintNode> duplicatedEntries = elementDeclarations.get(qName);

			// Comparing the first entry with all entries will imply
			// that the entries in the list are consistent
			for (int i = 1, duplicateCount = duplicatedEntries.size(); i < duplicateCount; i++)
			{
				if (!isEqual(duplicatedEntries.get(0), duplicatedEntries.get(i)))
				{
					return false;
				}
			}
		}
		return true;
	}


	private boolean isEqual(ConstraintNode targetSchemaNode1, ConstraintNode targetSchemaNode2)
	{
		return 	isEqual(targetSchemaNode1.getTargetElement(), targetSchemaNode2.getTargetElement()) &&
				isEqual(targetSchemaNode1.getTargetType(), targetSchemaNode2.getTargetType()) &&
				isEqual(targetSchemaNode1.getTargetRequired(), targetSchemaNode2.getTargetRequired());
	}


	private boolean isEqual(Object o1, Object o2)
	{
		return o1 == null ? o2 == null : o1.equals(o2);		
	}
}
