/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPathFunctionException;

import org.apache.xerces.dom.PSVIElementNSImpl;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.artifacts.EmbeddedTargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TargetInstanceBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TypeDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder.AnnotationResult;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.cosmos.rm.internal.validation.util.ParserHelper;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/**
 * This class is used to validate the sml:targetElement, sml:targetRequired and
 * sml:targetType constraints.
 *
 * @author sleeloy
 * @author Ali Mehregani
 * @author David Whiteman
 * @author John Arwe
 */
public class TargetValidator extends AbstractSMLValidator
{
	private Map<String, ConstraintNode> targetSchemaBuilderStructure;
	private Map<TargetElementNode, String> targetInstanceBuilderStructure;
	private DOMStructure documentDOMBuilderElems;
	private Map<String, Map<String, String>> substitutionElementElems;
	private ElementTypeMap elementTypeMapBuilderStructure;
	private TypeDeclarationBuilder typeDeclarationBuilder;
	private IValidationOutput<String, Object> validationLogger;



	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);

		DataBuilderRegistry builderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();

		builderRegistry.registerDataStructureBuilder(TargetInstanceBuilder.ID, new TargetInstanceBuilder());
		builderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());
		builderRegistry.registerDataStructureBuilder(TypeDeclarationBuilder.ID, new TypeDeclarationBuilder());
		builderRegistry.registerDataStructureBuilder(ElementDeclarationBuilder.ID, new ElementDeclarationBuilder());
	}

	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationTarget);
		validationLogger = getValidationOutput();
		documentDOMBuilderElems = (DOMStructure)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		TargetInstanceBuilder targetInstanceBuilder = (TargetInstanceBuilder) DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(TargetInstanceBuilder.ID);
		targetInstanceBuilderStructure = (Map<TargetElementNode, String>) targetInstanceBuilder.getDataStructure();
		targetSchemaBuilderStructure = (Map<String, ConstraintNode>) targetInstanceBuilder.getTargetSchemaBuilderStructure();
		substitutionElementElems = (Map<String, Map<String, String>>) targetInstanceBuilder.getSubstitutionGroupStructure();
		elementTypeMapBuilderStructure = targetInstanceBuilder.getElementTypeMap();
		typeDeclarationBuilder = (TypeDeclarationBuilder)DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(TypeDeclarationBuilder.ID);

		// Check the syntax first
		if (!isSyntaxValid())
		{
			if (shouldAbortOnError())
			{
				return false;
			}
		}

		// Validate the target constraints
		return validateValues() && validateTargetInstance();
	}


	/**
	 * Verifies that the values specified for sml:targetElement and sml:targetType
	 * are valid (i.e. The values exist in the scope of the document)
	 *
	 * @return true if validation passes; false otherwise
	 */
	private boolean validateValues()
	{
		for (Iterator<String> elementNames = targetSchemaBuilderStructure.keySet().iterator(); elementNames.hasNext();)
		{
			String elementName = elementNames.next();
			ConstraintNode schemaNode = targetSchemaBuilderStructure.get(elementName);

			QName targetElement = schemaNode.getTargetElement();
			if (targetElement != null && elementTypeMapBuilderStructure.getType(targetElement.getNamespaceURI(), targetElement.getLocalPart()) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetElementInvalidValue, targetElement.getPrefix() + ":" + targetElement.getLocalPart())));
				return false;
			}

			QName targetType = schemaNode.getTargetType();
			TypeDeclarationCollection typeCollection = typeDeclarationBuilder.getDataStructure();
			if (targetType != null && typeCollection.get(targetType) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalidValue, targetType.getPrefix() + ":" + targetType.getLocalPart())));
				return false;
			}
		}
		return true;
	}


	protected boolean validateTargetInstance()
	{
		Iterator<TargetElementNode> iter = targetInstanceBuilderStructure.keySet().iterator();

		// Check to make sure the type declarations are consistent
		if (!typeDeclarationBuilder.isStructureValid())
		{
			validationLogger.reportMessage(typeDeclarationBuilder.getErrorMessage());
			return false;
		}

		// For each element that has an associated target* constraint
		while (iter.hasNext())
		{
			TargetElementNode targetElementNode = (TargetElementNode)iter.next();
			ConstraintNode targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(targetElementNode.getName());

			// Check substitution group
			if (targetSchemaNode == null){
				Map<String, String> subGroupMap = substitutionElementElems.get(targetElementNode.getUri());
				if (subGroupMap != null){
					Object subGroup = subGroupMap.get(targetElementNode.getName());
					if (subGroup != null)
						targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(ParserHelper.removeNameSpace((String)subGroup));
				}
			}

			if (targetElementNode.isRequired()) // If SML reference was governed by an element declaration with sml:targetRequired=true
			{
				if (targetElementNode.isNilref()) { // URI resolution should not be attempted for nil references, and the constraint is automatically unsatisfied
					Object[] errorParams = new String[3];
					errorParams[0] = targetElementNode.getName();
					errorParams[1] = targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
					errorParams[2] = targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
					validationLogger.reportMessage(
							ValidationMessageFactory.createErrorMessage(
									targetElementNode.getLocation().getFilePath(),
									targetElementNode.getLocation().getLineNumber(),
									NLS.bind(SMLValidationMessages.unresolvedNilRef,errorParams)
									)
							);
					return false;
				} else if ( !isURISMLDeferenceable(targetElementNode)) {	// Attempt URI resolution for non-nil SML references
					Object[] errorParams = new String[3];
					errorParams[0] = targetElementNode.getName();
					errorParams[1] = targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
					errorParams[2] = targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
					validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.unresolvedTargetURI,errorParams)));
					return false;
				}
			}

			// check if we need to check the target element
			if (targetSchemaNode.getTargetElement() != null){ // should be using targetElementNode, not putative schema node cached using the local name of the SML reference element
				if (!validateTargetElement(targetElementNode, targetSchemaNode))
					return false;
			}
			if (targetSchemaNode.getTargetType() != null){ // should be using targetElementNode, not putative schema node cached using the local name of the SML reference element
				if (!validateTargetType(targetElementNode, targetSchemaNode))
					return false;
			}

		}
		return true;
	}

	protected boolean validateTargetType(TargetElementNode targetElementNode, ConstraintNode targetSchemaNode) {

		String targetElementNamespace = targetElementNode.getUri();
		String targetElementUri;
		try {
			targetElementUri = determineTargetElementURI(targetElementNode);
		} catch (URISyntaxException e) {
			return false;
		}

		// check if schema node matches instance node
		String getTargetType = elementTypeMapBuilderStructure.getType(targetElementNode.getUri(), targetElementUri);
		String targetSchemaNodeTypeName = qNameToString(targetSchemaNode.getTargetType());
		if ((getTargetType != null)){
			if (compareElementInstance(targetSchemaNodeTypeName, getTargetType)) {
				return true;
			}
			if (checkInheritanceTree(targetSchemaNodeTypeName, targetElementUri)) {
				return true;
			}
		}

		String targetTypeUri = null;
		// the ref could be an alias
		// Search through alias data structure for actual element
		Node alias = documentDOMBuilderElems.get(targetElementUri);
		if (alias != null) {
			targetTypeUri = alias.getNodeName();
			targetElementNamespace = alias.getNamespaceURI();
		}

		if (targetTypeUri != null) {
			getTargetType = elementTypeMapBuilderStructure.getType(targetElementNamespace, targetTypeUri);
		}
		if (getTargetType != null) {
			if (compareElementInstance(getTargetType, targetSchemaNodeTypeName)) {
				return true;
			}
			if (checkInheritanceTree(targetSchemaNodeTypeName, targetElementUri)) {
				return true;
			}
		}

		// there could be a substitution group
		// DLW - commenting out for now since this looks like it would have
		// caused a CCE
		// String substitutionGroup =
		// substitutionElementElems.get(targetElementUri);
		boolean valid = false;
		// while (substitutionGroup != null){
		// getTargetType =
		// elementTypeMapBuilderStructure.getType(targetElementUri
		// ,(String)substitutionGroup);
		// if (compareElementInstance(substitutionGroup,
		// qNameToString(targetSchemaNode.getTargetType())))
		// {
		// valid = true;
		// break;
		// }
		// substitutionGroup = substitutionElementElems.get(substitutionGroup);
		// }
		// //if dangling then it's okay
		if (valid || alias == null)
			return true;

		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;
		errorParams[4] = targetSchemaNodeTypeName;
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalid,errorParams)));

		return false;
	}

	/**
	 *
	 * @param targetSchemaNode
	 * @param targetElementUri
	 * @param getTargetType
	 * @return
	 */
	private boolean checkInheritanceTree(String targetSchemaNodeTypeName, String targetElementUri) {
		boolean valid = false;
		PSVIElementNSImpl psviElementNode = (PSVIElementNSImpl) documentDOMBuilderElems.get(targetElementUri);
		XSTypeDefinition type = (XSTypeDefinition) psviElementNode.getTypeDefinition();
		while ((type instanceof XSComplexTypeDefinition) && (((XSComplexTypeDefinition) type).getContentType() == XSComplexTypeDefinition.CONTENTTYPE_ELEMENT)) {
			if (compareElementInstance(type.getName(), targetSchemaNodeTypeName)) {
				valid = true;
				break;
			}
			type = type.getBaseType();
		}
		return valid;
	}

	protected boolean validateTargetElement(TargetElementNode targetElementNode, ConstraintNode targetSchemaNode) {

		String targetElementNamespace = targetElementNode.getUri();
		String targetElementUri;
		try {
			targetElementUri = determineTargetElementURI(targetElementNode);
		} catch (URISyntaxException e) {
			return false;
		}

		// check if schema node matches instance node
		if (compareElementInstance(qNameToString(targetSchemaNode.getTargetElement()), targetElementUri)) {
			return true;
		} else {
			// the ref could be an alias
			// Search through alias data structure for actual element
			Node alias = documentDOMBuilderElems.get(targetElementUri);
			if (alias != null){
				targetElementUri = alias.getNodeName();
				targetElementNamespace  = alias.getNamespaceURI();
			}
			if (compareElementInstance(targetElementUri, qNameToString(targetSchemaNode.getTargetElement()))){
				return true;
			}
			// there could be a substitution group
			Map<String, String> substitutionGroupMap = substitutionElementElems.get(targetElementNamespace);
			boolean valid = false;
			if (substitutionGroupMap != null){
				String substitutionGroup = substitutionGroupMap.get(targetElementUri);
				while (substitutionGroup != null){
					if (compareElementInstance(substitutionGroup, qNameToString(targetSchemaNode.getTargetElement()))){
						valid = true;
						break;
					}
					substitutionGroup = ParserHelper.removeNameSpace(substitutionGroup);
					substitutionGroup = substitutionGroupMap.get(substitutionGroup);
				}
			}
			// if dangling then it's okay
			if (valid || alias == null) return true;
		}
		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;

		QName targetElement = targetSchemaNode.getTargetElement();
		String prefix = targetElement.getPrefix();
		errorParams[4] = prefix != null && prefix.length() > 0 ? prefix + ":" + targetElement.getLocalPart() : targetElement.getLocalPart();

		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.invalidTargetInstance,errorParams)));
		return false;
	}

	private String determineTargetElementURI(TargetElementNode targetElementNode) throws URISyntaxException {
		String targetElementUri = targetElementNode.getTargetURI();

		try {
			Node contextNode = SMLValidatorUtil.locateDocumentNode(targetElementNode.getAliases(), targetElementNode.getDocumentPosition());
			URIReference targetURI = new URIReference(contextNode, targetElementUri, targetElementNode.getBaseURIValue());
			targetElementUri = targetURI.getDocumentReference();
		} catch (URISyntaxException exception) {
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), exception.getLocalizedMessage()));
			throw exception;
		} catch (Exception exc) {
			exc.printStackTrace();
		}
		return targetElementUri;
	}

	private String qNameToString(QName qName)
	{
		String prefix = qName.getPrefix();
		return prefix == null || prefix.length() <= 0 ? qName.getLocalPart() : prefix + ":" + qName.getLocalPart();
	}

	/**
	 * Check to make sure the syntax of the identity constraints
	 * is correct.  We need to use the XSModel object to retrieve
	 * all identity constraints.  The reason why the identity constraint
	 * structure can't be used is because there is no guarantee
	 * for all element declarations to have an instance (i.e. You
	 * can have an element declaration E with an invalid identity
	 * constraint that has no instance)
	 *
	 * @return true if the identity constraints are syntactically
	 * correct; false otherwise
	 */
	private boolean isSyntaxValid()
	{
		XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
		if (xsModel == null)
		{
			return true;
		}

		XSNamedMap typeDefinitions = xsModel.getComponents(XSConstants.TYPE_DEFINITION);
		boolean shouldAbortOnError = shouldAbortOnError();
		boolean status = true;

		// For every type definition, check for bad declarations
		for (int i = 0, definitionCount = typeDefinitions.getLength();
			 i < definitionCount /*&& (!shouldAbortOnError || status)*/; i++)
		{
			XSTypeDefinition typeDefinition = (XSTypeDefinition)typeDefinitions.item(i);
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE, false));
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE, false));
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE, false));
		}

		XSNamedMap elementDeclarations = xsModel.getComponents(XSConstants.ELEMENT_DECLARATION);
		// For every element declaration, check for bad values
		//	TODO Issue warning if target* constraint on element ref= (only allowed on name=)
		for (int i = 0, declarationCount = elementDeclarations.getLength();
			 i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSElementDeclaration elementDeclarationToCheck = (XSElementDeclaration)elementDeclarations.item(i);
			AnnotationResult targetElementAnnotationResult = SMLValidatorUtil.retrieveAnnotation(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
			AnnotationResult targetTypeAnnotationResult = SMLValidatorUtil.retrieveAnnotation(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
			status = checkForInvalidTargetElementValue(elementDeclarations, elementDeclarationToCheck, targetElementAnnotationResult);
			if (!shouldAbortOnError || status) {
				status = status && checkForInvalidTargetType(typeDefinitions, elementDeclarations, elementDeclarationToCheck, targetTypeAnnotationResult);
			}
		}
		// clear cache before starting the check for inconsistent declarations and invalid derivations

		if (!shouldAbortOnError || status)
		{
			status = status && checkForInvalidDerivations();
		}
		//	Check for consistency of substitution group specifications (another form of "derivation") 
		if (!shouldAbortOnError || status)
		{
			status = status && checkForInconsistentSubstitutionGroupSpecifications(elementDeclarations);
		}
		return status;
	}

	private boolean checkForInvalidTargetType(XSNamedMap allTypeDefs, XSNamedMap globalElementDeclarations, XSElementDeclaration elementDeclarationWithTargetType, AnnotationResult result) throws DOMException {
		boolean status = true;
		if (result != null) {
			Node[] nodes = result.getNodes();
			if (nodes.length > 0) {
				String targetTypeValue = nodes[0].getNodeValue();
				//	TODO change to use findTypeDef, which was built from this loop
				boolean found = false;
				XSTypeDefinition typeDef = null;
				for (int j = 0, typeDefinitionCount = allTypeDefs.getLength();
				 j < typeDefinitionCount && (!found); j++)
				{
					typeDef = (XSTypeDefinition) allTypeDefs.item(j);
					if (found = compareElementInstance(typeDef.getName(), targetTypeValue)) {
						break;
					}
				}
				if (!found) {
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.targetTypeInvalidValue, targetTypeValue)));
					if (shouldAbortOnError()) {
						return false;
					}
				}

			}
		}
		return status;
	}


	private XSTypeDefinition findTypeDef(XSNamedMap allTypeDefs,  String targetTypeValue) {
		if (allTypeDefs == null)
		{	//	Not provided as input, so find it
			XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
			if (xsModel == null)
			{
				return null;
			}
			allTypeDefs = xsModel.getComponents(XSConstants.TYPE_DEFINITION);
		}
		//	Now that the set of all type defs is in hand, search for the target
		boolean found = false;
		XSTypeDefinition typeDef = null;
		for (int j = 0, typeDefinitionCount = allTypeDefs.getLength();
		j < typeDefinitionCount && (!found); j++)
		{
			typeDef = (XSTypeDefinition) allTypeDefs.item(j);
			if (found = compareElementInstance(typeDef.getName(), targetTypeValue)) {
				//	Note that compareElementInstance compares only local names, not QNames
				break;
			}
		}
		return found == true ? typeDef : null;
	}


	protected boolean handleNewElement(XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition)
	{
		// We haven't cached this node yet (the caller checked), so do so.
		// Even if this one has no SML constraints, a later one (say the last one checked with the same Qname) might have an SML constraint.
		//	Unless we cache the "no constraints" case, the code would fail to flag such a case as inconsistent.
		String qName = SMLValidatorUtil.createQualifiedName(elementDeclaration);
		String currentElementTargetTypeUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
		String currentElementTargetElementUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
		String currentElementTargetRequired = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
		
		EmbeddedTargetElementNode embeddedTargetElementNode = new EmbeddedTargetElementNode(
				elementDeclaration, typeDefinition, 
				currentElementTargetTypeUri, currentElementTargetElementUri, currentElementTargetRequired );
		getConstraintMap().put(qName, embeddedTargetElementNode);
		return true;
	}


	protected boolean handleDiscoveredElement(XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition, Object discoveredElement)
	{
		//	"Retrieve" the (possibly null) target* constraint values from the first same-Qnamed element declaration
		EmbeddedTargetElementNode cachedElement = (EmbeddedTargetElementNode)discoveredElement;
		
		//	Retrieve the (possibly null) target* constraint values from the current (n-th) same-Qnamed element declaration
		String currentElementTargetTypeUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
		String currentElementTargetElementUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
		String currentElementTargetRequired = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
		
		//	SML constraints http://www.w3.org/TR/sml/#Constraints_on_Targets talks about about
		//	same-named elements in a type being brought in directly, implicitly, or indirectly.
		//	Those refer to coding name=X twice, model groups, and substitution groups.
		//	Derivation is orthogonal and covered later in 5.4 http://www.w3.org/TR/sml/#Sml_Constraints_and_Derivation_Overview
		
		// target type should be checked to see if it's the same hierarchy, because if it's not,
		// there is no error
		//	TODO Manage each set of related types enclosed a given element decl name separately.
		//	The existing comment (that I believe is correct) I think is trying to say that this consistency
		//	checking for same-named element declarations only should occur amongst element declarations
		//	contained in the same type definition.  That is, an element decl A that is contained in two
		//	unrelated complex types (CT1, CT2) need not have consistent SML constraints.  That in turn
		//	means that the implementation choice to index the hash map only by element Qname carves
		//	up the world incorrectly (it lumps all like-named element decls together, whether or not their
		//	enclosing complex types are related).  More structure is needed to group the subsets of like-named
		//	element decls that are related via their inclusion in complex types related by inheritance.
		//	Might be able to deal with this by iterating (at the root level, in the Abstract validator class)
		//	only over those CTs that are not derived from others.
		if (currentElementTargetTypeUri != null)
		{
			if (!checkForSameHierarchy(typeDefinition, cachedElement.getEnclosingType()))	//	Also checks that no restriction is used
			{
				return true;
			}
		}
		
		//	Set the default error message string.
		String errorMessage;
		// If they are the same exact complex type definition, then this is a situation where two element nodes
		// that aren't identical but have the same name are enclosed in the same complex type,
		// either directly or indirectly.  Otherwise they are related through derivation.
		if (typeDefinition == cachedElement.getEnclosingType()) {
			errorMessage = SMLValidationMessages.targetInconsistentDeclarations;
		}
		else
		{
			errorMessage = SMLValidationMessages.targetBadDerivation;
		}
		// TODO We have 2 decls in hand being compared.  Include their names, enclosing type names, and which target* is bad in output.

		//	---------------------- Check consistency of target types
		if (currentElementTargetTypeUri != null && cachedElement.getTargetType() != null)
		{
			String cachedElementTargetTypeUri =  cachedElement.getTargetType().toString();
			if (currentElementTargetTypeUri.equals(cachedElementTargetTypeUri))
			{
				//	Identical, clearly compatible regardless of the particular case involved.
			}
			else if (typeDefinition == cachedElement.getEnclosingType()) // same-named element decl contained in 1 type, must be identical
			{
				//	Type strings are different.  As noted above, this could be due to conditions like different 
				//	namespace prefixes for the same namespace URI being used on the values.
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
				return false;
			}
			else //	TODO Remove simplifying assumption about target namespaces 
			{
				//	In order to test equality of types (which are specified as QNames), need the ability to resolve namespace
				//	prefixes in the context of the schema document.  The current design makes that difficult.
				//	As a simplifying assumption, assume a single target namespace and search by local name.
				XSTypeDefinition tdCurrent = findTypeDef(null , ParserHelper.removeNameSpace(currentElementTargetTypeUri) );
				XSTypeDefinition tdCached = findTypeDef(null , cachedElementTargetTypeUri );	//	ns already set to null by cachedElement constructor
				//	Since the model was already checked for invalid target type values, neither result will be null.
				if (isRestrictionOf(typeDefinition , cachedElement.getEnclosingType())) 
				{
					// current is a restriction of cached; since not equal, only valid if same relationship holds between target types 
					if (!isRestrictionOf( tdCurrent , tdCached))
					{
						validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
						return false;
					}
				}
				else if (isRestrictionOf(cachedElement.getEnclosingType() , typeDefinition )) 
				{
					// cached is a restriction of current; since not equal, only valid if same relationship holds between target types 
					if (!isRestrictionOf( tdCached , tdCurrent))
					{
						validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
						return false;
					}
				}
	
			}
		}
		else if (currentElementTargetTypeUri == null && cachedElement.getTargetType() == null)
		{
			//	consistent by definition, so continue
		}
		else
		{	//	One has it, the other does not, so inconsistent
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
			return false;	
		}

		//	------------------------------ Check consistency of target elements
		//	TODO There might be a need to handle restriction specially, but it's not clear what that concept means for elements.
		//	Conceptually one could treat a substitution group affiliation as a case of derivation, with the SG head playing the role of ancestor.
		if (currentElementTargetElementUri != null && cachedElement.getTargetElement() != null)
		{
			String cachedElementTargetElementUri =  cachedElement.getTargetElement().toString();
			if (! compareElementInstance( currentElementTargetElementUri , cachedElementTargetElementUri))
			{
				//	Type strings are different.  As noted above, this could be due to conditions like different 
				//	namespace prefixes for the same namespace URI being used on the values.
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
				return false;
			}
		}
		else if (currentElementTargetElementUri == null && cachedElement.getTargetElement() == null)
		{
			//	consistent by definition, so continue
		}
		else
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
			return false;	//	One has it, the other does not, so inconsistent
		}


		//	------------------------------ Check consistency of target required
		if (currentElementTargetRequired != null )	//	specified, but still might be true or false
		{
			if (SMLValidatorUtil.isTrue(currentElementTargetRequired) == cachedElement.getTargetRequired())
			{
				//	Same, so consistent 
			}
			else if (typeDefinition == cachedElement.getEnclosingType()) 
			{
				// Same-named element decl contained in 1 type, must be identical 
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
				return false;
			}
			else if (isRestrictionOf(typeDefinition , cachedElement.getEnclosingType())) 
			{
				// current is a restriction of cached; since not equal, only valid if current=true and cached (ancestor) = false 
				if (!(SMLValidatorUtil.isTrue(currentElementTargetRequired) && !cachedElement.getTargetRequired()))
				{
					validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
					return false;
				}
			}
			else if (isRestrictionOf(cachedElement.getEnclosingType() , typeDefinition )) 
			{
				// cached is a restriction of current; since not equal, only valid if cached=true and current (ancestor) = false 
				if (!(SMLValidatorUtil.isFalse(currentElementTargetRequired) && cachedElement.getTargetRequired()))
				{
					validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
					return false;
				}
			}
		}
		else if (cachedElement.getTargetRequired())	//	current=unspecified (defaults to false), cached one was =true 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
			return false;	//	One has it, the other does not, so inconsistent
		}

		// if either targetElement or targetType exists, we should make sure it doesn't
		// equal the value of the cached node that has the same name
//		if (targetUri != null) {
//			String errorMessage = SMLValidationMessages.targetBadDerivation;
//			// If they are the same exact class, then this is a situation where two element nodes
//			// that aren't identical but have the same name are enclosed in the same complex type,
//			// either directly or indirectly.
//			if (typeDefinition == elementNameConstraints.getEnclosingType()) {
//				errorMessage = SMLValidationMessages.targetInconsistentDeclarations;
//			}
//			if (!targetUri.equals(elementNameConstraints.getTargetURI())) {
//				// TODO do we need file path info here?  if so, where do we get it from?
//				// TODO Also would be good to have accurate location info
//				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
//				return false;
//			}
//			if ((currentElementTargetRequired != null) && (SMLValidatorUtil.isTrue(currentElementTargetRequired) != elementNameConstraints.isRequired())) {
//				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
//				return false;
//			}
//		}

		return true;
	}


	/**
	 * Answer the value of an attribute for a given declaration.
	 *
	 * @param declaration
	 * @param namespaceUri
	 * @param attributeName
	 * @return
	 */
	protected String retrieveAnnotationValue(XSElementDeclaration declaration, String namespaceUri, String attributeName) {
		String result = null;
		AnnotationResult annotation = SMLValidatorUtil.retrieveAnnotation(declaration, namespaceUri, attributeName);
		if ((annotation != null) && (annotation.getNodes().length > 0)) {
			return annotation.getNodes()[0].getNodeValue();
		}
		return result;
	}

	private boolean checkForSameHierarchy(XSComplexTypeDefinition parentComplexTypeForCurrentElement, XSComplexTypeDefinition enclosingType) {
		return ((parentComplexTypeForCurrentElement.derivedFromType(enclosingType, XSConstants.DERIVATION_EXTENSION)) || (enclosingType.derivedFromType(parentComplexTypeForCurrentElement, XSConstants.DERIVATION_EXTENSION)));
	}

	private boolean isRestrictionOf(XSTypeDefinition child , XSTypeDefinition parent) {
		return ((child.derivedFromType(parent, XSConstants.DERIVATION_RESTRICTION)) );
	}

	private void checkForBadDeclaration(AnnotationResult annotationResult) {
		if (annotationResult != null) {
			getValidationOutput().reportMessage(ValidationMessageFactory.createWarningMessage(
					ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.targetBadDeclaration));
		}
	}

	//	Removes any namespace prefixes and compares only the local name portions of the input strings
	protected boolean compareElementInstance(String elem1, String elem2)
	{
		return (ParserHelper.removeNameSpace(elem1).equals(ParserHelper.removeNameSpace(elem2)));
	}

	private boolean checkForInvalidTargetElementValue(XSNamedMap elementDeclarations, XSElementDeclaration elementDeclaration, AnnotationResult result) throws DOMException {
		boolean status = true;
		if (result != null) {
			Node[] nodes = result.getNodes();
			if (nodes.length > 0) {
				String targetElementValue = nodes[0].getNodeValue();
				boolean found = false;
				for (int j = 0, declarationCount2 = elementDeclarations.getLength();
				 j < declarationCount2 && (!found); j++)
				{
					XSElementDeclaration elemDec2 = (XSElementDeclaration) elementDeclarations.item(j);
					if (elemDec2 != elementDeclaration) {
						found = compareElementInstance(elemDec2.getName(), targetElementValue);
					}
				}
				if (!found) {
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.targetElementInvalidValue, targetElementValue)));
					status = false;
				}
			}
		}
		return status;
	}

	/**
	 * Make sure that the uri can be resolved.
	 * @param uri the uri string to resolve
	 * @return true if the uri can be resolved otherwise false.
	 */
	protected static boolean isURISMLDeferenceable(TargetElementNode targetElementNode)
	{
		try
		{
			String uri = targetElementNode.getTargetURI();
			if ((uri == null) || "".equals(uri.trim())) {
				return false;
			}
			Node context = SMLValidatorUtil.locateDocumentNode(targetElementNode.getAliases(), targetElementNode.getDocumentPosition());
			return (((DerefXPathFunction)DerefXPathFunction.instance()).evaluate(context, SMLValidatorUtil.removeLineBreaks(uri), targetElementNode.getBaseURIValue()) != null);
		}
		catch (XPathFunctionException e)
		{
			return false;
		}
	}

	//	
	private boolean checkForInconsistentSubstitutionGroupSpecifications(XSNamedMap elementDeclarations) 
	{
		boolean shouldAbortOnError = shouldAbortOnError();
		boolean status = true;
		for (int i = 0, declarationCount = elementDeclarations.getLength();
		i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSElementDeclaration elementDeclarationToCheck = (XSElementDeclaration)elementDeclarations.item(i);
			XSElementDeclaration sg = elementDeclarationToCheck.getSubstitutionGroupAffiliation();
			if (sg != null)
			{
				//	This is an approximation of what is required.  A complete implementation would require
				//	following the entire SG chain (separately, for each constraint) until an explicit specification 
				//	or the end of the SG chain is reached.  E.g. to cover ED1(TT=T1,TE=EDx,SG=ED2) ED2=(TT=T2,SG=ED3) ED3=(TE=EDy) 
				String targetElementAR = retrieveAnnotationValue(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
				if (targetElementAR != null)
				{
					String targetElementSGAR = retrieveAnnotationValue(sg, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
					if (targetElementSGAR != null)
					{
						// TODO are target element values compatible? http://www.w3.org/TR/sml/#Target_Schema_Validity_Rules clause 1
					}
				}
				
				String targetTypeAR = retrieveAnnotationValue(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
				if (targetTypeAR != null)
				{
					String targetTypeSGAR = retrieveAnnotationValue(sg, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
					if (targetTypeSGAR != null)
					{
						// TODO are target type values compatible? http://www.w3.org/TR/sml/#Target_Schema_Validity_Rules clause 1
					}
				}

				String targetRequiredAR = retrieveAnnotationValue(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
				if (targetRequiredAR != null)
				{
					String targetRequiredSGAR = retrieveAnnotationValue(sg, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
					if (targetRequiredSGAR != null)
					{
						if (SMLValidatorUtil.isFalse(targetRequiredAR) && SMLValidatorUtil.isTrue(targetRequiredSGAR) )		// SG head more stringent than its replacement 
						{
							validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, 
									NLS.bind(SMLValidationMessages.targetBadSubstitution, elementDeclarationToCheck.getName())));
							status = false;
							if (shouldAbortOnError)
							{
								return status;
							}
						}
					}
				}
				
			}	//	Current element has a substitution group affiliation
		}	//	Check next element declaration
		return status;
	}
}

