/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.common;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.cosmos.rm.internal.smlif.SMLPlugin;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.XMLInternalUtility;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Provides common utility methods that used by multiple classes
 * under different packages.
 * 
 * @author Ali Mehregani
 */
public class SMLCommonUtil
{	

	/**
	 * Retrieve the child node of 'parent' with the uri and localName passed in
	 * 
	 * @param parent The parent node
	 * @param uri The URI
	 * @param localName The local name of the element
	 * @return The element node or null if it cannot be found
	 */
	public static Node retrieveChildNode(Node parent, String uri, String localName)
	{
		if (parent == null)
			return null;
		
		NodeList children = parent.getChildNodes();		
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			if (uri.equals(child.getNamespaceURI()) && localName.equals(child.getLocalName()))
			{
				return child;				
			}
		}
		
		return null;
	}

//	/**
//	 * Retrieve the child node of 'parent' with the uri and localName passed in
//	 * 
//	 * @param parent The parent node
//	 * @param uri The URI of the attribute
//	 * @param localName The local name of the attribute
//	 * @return The attribute value or empty string if the attribute cannot be found
//	 */
//	public static String retrieveAttributeNode(Node parent, String uri, String localName)
//	{
//		if (parent == null)
//			return "";
//		
//		NamedNodeMap attributes = parent.getAttributes();
//		int count = attributes.getLength();
//		for(int idx=0;idx<count;idx++)
//		{
//			Node node = attributes.item(idx);
//			if(node != null && node.getNamespaceURI() != null
//					&& node.getNamespaceURI().equals(ISMLConstants.XML_URI)
//					&& node.getLocalName() != null && node.getLocalName().equals(ISMLConstants.BASE_URI_ELEMENT))
//			{
//				return node.getNodeValue();
//			}
//		}
//		return "";
//	}
	
	
	/**
	 * Retrieve the text child node of the argument node passed
	 * 
	 * @param node The node
	 * @return The child text of node
	 */
	public static String retrieveTextNode(Node node)
	{
		if (node == null)
			return null;
		
		StringBuffer sb = new StringBuffer();
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node textNode = children.item(i);
			String currentText = null;
			if (textNode.getNodeType() == Node.TEXT_NODE && (currentText = textNode.getNodeValue()) != null
					&& (currentText = currentText.trim()).length() > 0)
			{
				sb.append(currentText);
			}
		}
		
		/* Walk through the node text and remove new lines and unnecessary white spaces */
		StringTokenizer st = new StringTokenizer(sb.toString(), ISMLConstants.nl);
		StringBuffer finalTextValue = new StringBuffer();
		while (st.hasMoreTokens())
		{
			String currentLine = st.nextToken().trim();
			if (currentLine.length() <= 0)
				continue;
			
			if (finalTextValue.length() > 0)
				finalTextValue.append(" "); //$NON-NLS-1$
			finalTextValue.append(currentLine);			
		}
		
		return finalTextValue.toString();
	}

	

	/**
	 * Equivalent to nestedNodeRetrieval (node, nodeNames, false, true)
	 */
	public static Node nestedNodeRetrieval(Node node, String[][] nodeNames)
	{
		return nestedNodeRetrieval (node, nodeNames, false, true, null);
	}
	
	
	/**
	 * Retrieve the nested node as specified by nodeNames (starting from 'node').
	 * The nodeNames are expected to be any array of String[] where String[0] = 
	 * URI of a node and String[1] = localName.
	 * 
	 * For example, given:
	 * <pre>
	 * 	<a><b><c></c></b></a>
	 * </pre>
	 * The node 'c' can be retrieved using
	 * nestedNodeRetrieval(node, new String[][] {new String[]{b's URI, "b"}, new String[]{c's URI, "c"}});
	 * where node corresponds to a's node.
	 * 	
	 * @param node The starting node
	 * @param nodeNames The node names
	 * @param create Creates the node if none is found
	 * @param siblingToInsertBefore The sibling to insert before
	 * @return The node representing the last entry of nodeNames or null if none 
	 * is found
	 */
	public static Node nestedNodeRetrieval(Node node, String[][] nodeNames, boolean create, boolean append, Node siblingToInsertBefore)
	{
		Node currentNode = node;
		for (int i = 0; i < nodeNames.length; i++)
		{
			if (currentNode == null)
				return null;
			
			Node childNode = retrieveChildNode(currentNode, nodeNames[i][0], nodeNames[i][1]);
			if (childNode == null && create)
			{
				childNode = createFormattedElement(currentNode, nodeNames[i][0], nodeNames[i][1], append, siblingToInsertBefore);
			}
			currentNode = childNode;			
		}
		
		return currentNode;
		
	}

	public static Node createElement (Node parentNode, String uri, String localName)
	{
		Node node = parentNode.getOwnerDocument().createElementNS(uri, localName);
		node.setPrefix(parentNode.getPrefix());
		parentNode.appendChild(node);
		
		return node;
	}

	public static Node createFormattedElement(Node parentNode, String uri, String localName, boolean append, Node siblingToInsertBefore)
	{
		Node childNode = null;
		Document ownerDocument = parentNode.getOwnerDocument();
		childNode = ownerDocument.createElementNS(uri, localName);
		childNode.setPrefix(parentNode.getPrefix());
//		Node firstChild = parentNode.getFirstChild();
		
//		if (siblingToInsertBefore == null) {
//			siblingToInsertBefore = firstChild;
//		}
		
		/* Find out what the indent level of the last element is */
		Node sibling = append ? parentNode.getLastChild() : siblingToInsertBefore;
		Node indentNode = null;
		while (sibling != null && sibling.getNodeType() != Node.ELEMENT_NODE)
		{
			sibling = sibling.getPreviousSibling();
		}				
		if (sibling != null && sibling.getNodeType() == Node.ELEMENT_NODE)
		{
			indentNode = sibling.getPreviousSibling();
			if (XMLInternalUtility.isWhitespace(indentNode))
			{
				/* Remove the last line entry */
				if (append) {
					Node lastChildNode = parentNode.getLastChild();
					if (XMLInternalUtility.isWhitespace(lastChildNode))
						parentNode.removeChild(lastChildNode);
				}
				
				Node indent = ownerDocument.createTextNode(indentNode.getNodeValue());
				if (append)
				{
					parentNode.appendChild(indent);
				}
				else
				{
					parentNode.insertBefore(indent, siblingToInsertBefore);
				}
			}
		}
		
		/* Add the child node followed by an indent/linebreak */
		if (append)
			parentNode.appendChild(childNode);
		else
		{
			if (indentNode != null)
			{
				parentNode.insertBefore(childNode, indentNode.getNextSibling());
			}
			else
			{
				parentNode.insertBefore(childNode, siblingToInsertBefore);
			}			
		}
		
		Node parentIndent = parentNode.getPreviousSibling();		
		if (append && XMLInternalUtility.isWhitespace(parentIndent))
		{
			parentNode.insertBefore(ownerDocument.createTextNode(parentIndent.getNodeValue()), childNode.getNextSibling());	
		}	
		
		return childNode;
	}

	
	/**
	 * Retrieves and returns the aliases associated with document
	 * 
	 * @param userDefinedAliases The user defined aliases
	 * @param document The document
	 * 
	 * @return The aliases associated with document
	 */
	public static String[] retrieveAliases(Map<String,String[]> userDefinedAliases, ISMLDocument document)
	{
		String[] userAliases = (String[])userDefinedAliases.get(document.getMetadata().getId());
		userAliases = userAliases == null ? new String[0] : userAliases;
		
		String[] fileAliases = document.getMetadata().getAliases();
		fileAliases = fileAliases == null ? new String[0] : fileAliases;
		String[] finalAliases = new String[userAliases.length + fileAliases.length];
		System.arraycopy(userAliases, 0, finalAliases, 0, userAliases.length);
		System.arraycopy(fileAliases, 0, finalAliases, userAliases.length, fileAliases.length);
		
		return finalAliases;
	}


	/**
	 * Add a validator to the input page to ensure only valid URIs
	 * are input to the Text field.
	 * 
	 * @param textField
	 * @param page
	 */
	public static void addURIValidator(final Text textField, final SMLPage page) {
		textField.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				String baseURI = textField.getText();
				
				if (baseURI.equals("")) { //$NON-NLS-1$
					return;
				}
				String message = null;
				try {
					new URI(baseURI);
				} catch (URISyntaxException e1) {
					message = NLS.bind(SMLMessages.errorInvalidBaseURIFormat, baseURI);
				}
				page.setErrorMessage(message);
			}
		});
	}
	
	public static void openErrorWithDetail(Shell parent, String title, String message, String cause)
	{
		
		class CustomErrorDialog extends ErrorDialog
		{
			private String cause;
			private static final int LIST_ITEM_COUNT = 7;
			private org.eclipse.swt.widgets.List list;
			private boolean isListDisplayed;
			private Clipboard clipboard;
			
			public CustomErrorDialog(Shell parentShell, int severity, String dialogTitle, String message, String cause) 
			{
				super (parentShell, dialogTitle, (cause == null) ? null : message, new Status(severity, SMLActivator.PLUGIN_ID, severity, message, (cause == null) ? null : new Throwable(cause)), IStatus.OK | IStatus.INFO | IStatus.WARNING | IStatus.ERROR);
				this.cause = cause;
				isListDisplayed = false;
			}
			
			public void openErrorDialog()
			{
				this.open();
			}			
			
		    			
		    protected org.eclipse.swt.widgets.List createDropDownList(Composite parent) 
		    {
		    	if (isListDisplayed)
		    	{
		    		isListDisplayed = false;		    		
		    		list.dispose();
		    		return list;
		    	}
		    	
		    	isListDisplayed = true;
		    	
		    	// create the list
		    	list = new org.eclipse.swt.widgets.List(parent, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI);
	
		    	// fill the list
		    	if (cause != null)
		    	{
			    	StringTokenizer st = new StringTokenizer (cause, ISMLConstants.nl);
			    	String indents = "";
			    	boolean addedIndentation = false;
			    	while (st.hasMoreTokens())
			    	{
			    		String currentToken = st.nextToken();
			    		currentToken = currentToken.replace ('\r', ' ');
			    		currentToken = currentToken.trim();
			    		if (!addedIndentation && currentToken.startsWith("at"))
			    		{
			    			indents += "  ";
			    			addedIndentation = true;
			    		}
			    		else if (addedIndentation && !currentToken.startsWith("at"))
			    			addedIndentation = false;
			    		
			    		
			    		list.add(indents + currentToken);
			    	}
		    	}
		    	
		        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL
		                | GridData.GRAB_HORIZONTAL | GridData.VERTICAL_ALIGN_FILL
		                | GridData.GRAB_VERTICAL);
		        data.heightHint = list.getItemHeight() * LIST_ITEM_COUNT;
		        data.horizontalSpan = 2;
		        list.setLayoutData(data);
		        list.setFont(parent.getFont());
		        
		        Menu copyMenu = new Menu(list);
		        MenuItem copyItem = new MenuItem(copyMenu, SWT.NONE);
		        copyItem.addSelectionListener(new SelectionListener() {
		            /*
		             * @see SelectionListener.widgetSelected (SelectionEvent)
		             */
		            public void widgetSelected(SelectionEvent e) {
		                copyToClipboard();
		            }
	
		            /*
		             * @see SelectionListener.widgetDefaultSelected(SelectionEvent)
		             */
		            public void widgetDefaultSelected(SelectionEvent e) {
		                copyToClipboard();
		            }
		        });
		        
		        copyItem.setText(SMLValidationMessages.commonCopy);
		        list.setMenu(copyMenu);
		    	
		        return list;
		    }
		    
		    private void copyToClipboard() {
		        if (clipboard != null)
		            clipboard.dispose();
		        StringBuffer clipBoardBuffer = new StringBuffer();		       
		        
		        String[] listEntrySelections = list.getSelection();
		        for (int i = 0; i < listEntrySelections.length; i++)
		        	clipBoardBuffer.append(listEntrySelections[i] + IValidationConstants.LINE_SEPARATOR);
		        clipboard = new Clipboard(list.getDisplay());
		        clipboard.setContents(new Object[] { clipBoardBuffer.toString() },
		                new Transfer[] { TextTransfer.getInstance() });
		    }
	
	
		}
				
		CustomErrorDialog errorDialog = new CustomErrorDialog (parent, IStatus.ERROR, title, message, cause);
		errorDialog.openErrorDialog();
	}


	public static void openErrorWithDetail(final String title, final String message, Throwable t)
	{
		openErrorWithDetail(title, message, t, true);
	}

	public static void openErrorWithDetail(final String title, final String message, Throwable t, final boolean defaultParent)
	{	
		Throwable exceptionCause = null;
		String stackTrace = null;
		
		if (t != null) {
			stackTrace = SMLValidatorUtil.getExceptionStackTrace(t);
			exceptionCause = (t.getCause() != null ? t.getCause() : t);
		}
			
		final String cause = (exceptionCause == null) ? null : exceptionCause.getClass().getName() + ISMLConstants.nl + exceptionCause.getMessage() + ISMLConstants.nl + stackTrace;
		
		SMLPlugin.getDefault().getWorkbench().getDisplay().syncExec(new Runnable()
		{
	
			public void run() 
			{
				openErrorWithDetail(defaultParent ? SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell() : null, title, message, cause);
				
			}});
		
	}
		

}
