/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.common.validation;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;

import org.eclipse.cosmos.me.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;

/**
 * 
 * @author Eric S. Rose (esrose@us.ibm.com)
 *
 */
public interface XML_DAS {
	
	/**
	 * Create a Document representing the XML in the given InputStream.
	 * Checks if the input XML is well-formed but does NOT perform Schema validation.
	 * 
	 * @param inputStream InputStream from which to load the Document
	 * @param schemaFile Schema
	 * 
	 * @return Document representing associated XML
	 * 
	 * @throws IOException if there's a problem reading from the InputStream.
	 * @throws XMLValidationException if the XML in the InputStream is not well-formed.
	 */
	Document loadDocument(InputStream inputStream) throws IOException, XMLValidationException;
		
	/**
	 * Save the given Document to the given OutputStream and optionally performs validation.
	 * 
	 * @param document Document to save to XML
	 * @param outputStream OutputStream to which to write the DataGraph
	 * @param encoding The desired XML encoding of the resulting document  (null to use the default encoding)
	 * @param performValidation Whether or not to perform validation on resulting XML
	 * 
	 * @throws IOException if there's a problem writing to the OutputStream
	 * @throws XMLValidationException if the Document isn't valid against the associated set of Rules.
	 */
	void saveDocument(Document document, OutputStream outputStream, String encoding, boolean performValidation)
			throws XMLValidationException, IOException;
	
	/**
	 * Run all validation rules associated with the DAS instance.
	 *
	 * @param document Document to validate
	 * 
	 * @return A Collection of XMLValidationErrors, or an empty array if no error occurred
	 */
	Collection<XMLValidationError> validate(Document document);
	
	/**
	 * Add a ValidationRule to the list maintained by the DAS instance.
	 * 
	 * @param validationRule ValidationRule to add to the list.
	 */
	void addValidation(ValidationRule validationRule);
	
	/**
	 * Remove a ValidationRule from the validation list.
	 * Fails silently if the given ValidationRule is not in the list or the Rule list is empty.
	 * 
	 * @param validationRule ValidationRule to remove from the list.
	 */
	void removeValidation(ValidationRule validationRule);
	
	/**
	 * Get the list of ValidationRules associated with the DAS instance.
	 * 
	 * @return A Collection containing the ValidationRules associated with the DAS
	 */
	Collection<ValidationRule> getValidationRules();
}