/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.impl;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfMessages;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility.NodeXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.cmdbfservice.AbstractServiceHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.ICMDBfQueryOperation;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IItemConstraintHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IItemTemplateHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IQueryHandlerFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IRelationshipConstraintHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IRelationshipTemplateHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.QueryInputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.artifacts.IContentSelector;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.artifacts.ITemplate;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IDepthLimit;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IInstanceIdConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IItemReference;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IItemTemplate;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IPropertyId;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IQuery;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IQueryInputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRecordConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRelationshipEndpoint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.ISelectedRecordType;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IXPathExpression;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IEdges;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryServiceElementCollection;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.TransformerException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * This is the default implementation of {@link ICMDBfQueryOperation}.  Contributors
 * can choose to use this default implementation or provide a direct implementation
 * of {@link ICMDBfQueryOperation}.
 * 
 * 
 * @author Ali Mehregani
 */
public class CMDBfQueryOperation extends AbstractServiceHandler implements ICMDBfQueryOperation
{
	/**
	 * The handler factory class associated with this operation
	 */
	private IQueryHandlerFactory handlerFactory;
	
	/**
	 * The factory used for creating query input artifacts for the operation's repository 
	 */
	private IQueryInputArtifactFactory queryInputArtifactFactory;

	/**
	 * The factory used for creating query output artifacts for the operation's repository 
	 */
	private IQueryOutputArtifactFactory queryOutputArtifactFactory;

	/**
	 * The constructor.  This CMDBf query operation takes in a handler
	 * factory that will be used to construct constraint handlers when
	 * processing CMDBf queries. 
	 * 
	 * @param handlerFactory The handler factory associated with this
	 * operation
	 */
	public CMDBfQueryOperation(IQueryHandlerFactory handlerFactory, IQueryInputArtifactFactory queryInputArtifactFactory)
	{
		this.handlerFactory = handlerFactory;
		this.queryInputArtifactFactory = queryInputArtifactFactory;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.ICMDBfQueryOperation#execute(java.net.URI, java.io.InputStream)
	 */
	public IQueryResult execute(InputStream query) throws CMDBfServiceException
	{
		return execute((Object)query);	
	}

	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.ICMDBfQueryOperation#execute(java.io.Reader)
	 */
	public IQueryResult execute(Reader query) throws CMDBfServiceException
	{
		return execute((Object)query);	
	}
		
	private IQueryResult execute(Object input) throws CMDBfServiceException
	{		 
		try
		{
			// 1. Transform the query
			IQuery cmdbfQuery = input instanceof Reader ? 
						QueryInputTransformer.transform((Reader)input, queryInputArtifactFactory, getRuntimeOptions()) :
						QueryInputTransformer.transform((InputStream)input, queryInputArtifactFactory, getRuntimeOptions());
						
			// 2. Process the item templates
			IQueryResult queryResult = processItemTemplates(cmdbfQuery);
			
			// 3. Process the relationship templates
			processRelationshipTemplates(cmdbfQuery, queryResult);
			
			// 4. Process the content selectors
			processContentSelectors (queryResult, cmdbfQuery);
			return queryResult;
		} 
		catch (TransformerException e)
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER, 
					CMDBfServiceException.QUERY_ERROR, 
					e.getMessage());
		}		
	}
	
	
	private IQueryResult processItemTemplates(IQuery query) throws CMDBfServiceException
	{
		IQueryResult queryResult = getQueryOutputArtifactFactory().createQueryResult();		
		if (query.getItemTemplateCount() <= 0)
		{
			return queryResult;
		}		
				
		// Walk through each item template and apply its constraints
		IItemTemplate[] itemTemplates = query.getItemTemplates();
		IItemTemplateHandler itemHandler = handlerFactory.createItemTemplateHandler();
		if (!itemHandler.isInitialized())
		{
			itemHandler.initialize(getInit());
		}
		
		for (int i = 0; i < itemTemplates.length; i++)
		{						 					
			INodes result = itemHandler.execute(itemTemplates[i]);			
			IConstraint[] xpathExpression = itemTemplates[i].getXpathExpressions();
			
			// Apply constraints
			if (xpathExpression == null || xpathExpression.length == 0)
			{
				// 1. Apply the instance id constraints			
				IInstanceIdConstraint instanceConstraint = itemTemplates[i].getInstanceIdConstraint();
				if (instanceConstraint != null)
				{				
					result = applyItemConstraints (result, new IConstraint[]{instanceConstraint});
					if (result.isEmpty())
					{
						continue;
					}
				}
							
											
				// 2. Apply the record constraints
				IRecordConstraint[] recordConstraints = itemTemplates[i].getRecordConstraints();
				if (recordConstraints != null)
				{
					// For each record constraint
					for (int j = 0; j < recordConstraints.length; j++)
					{
						// Apply the record type constraint
						result = applyItemConstraints (result, recordConstraints[j].getRecordTypes());
						if (result.isEmpty())
						{
							break;
						}
						
						// Apply the property value constraint
						result = applyItemConstraints (result, recordConstraints[j].getPropertyValues());
						if (result.isEmpty())
						{
							break;
						}
					}
					
					if (result.isEmpty())
					{
						continue;
					}
				}
			}
			// Apply the XPath constraints
			else
			{							
				result = applyItemConstraints(result, itemTemplates[i].getXpathExpressions());					
			}
			
			
			if (!result.isStartingContext() && !result.isEmpty() && result.getType() == IQueryServiceElementCollection.NODES)
			{
				queryResult.addNodes((INodes)result);
			}
		}		
		return queryResult;
	}
	

	private void processRelationshipTemplates(IQuery cmdbfQuery, IQueryResult queryResult) throws CMDBfServiceException
	{
		
		IRelationshipTemplate[] relationshipTemplates = cmdbfQuery.getRelationshipTemplates();
		IRelationshipTemplateHandler relationshipHanlder = handlerFactory.createRelationshipTemplateHandler();
		if (!relationshipHanlder.isInitialized())
		{
			relationshipHanlder.initialize(getInit());
		}
		
		// This variable is used to store the revised list of items that should end up under items
		Map<String, List<IGraphElement>> items = new Hashtable<String, List<IGraphElement>>();
		
		// For every relationship template
		for (int i = 0; i < relationshipTemplates.length; i++)
		{	
			IEdges edges = getQueryOutputArtifactFactory().createEdges(relationshipTemplates[i]);
			INodes sourceNodes = queryResult.getNodes(retrieveId(relationshipTemplates[i].getSourceElement(), CMDBfMessages.relationshipSource));
			INodes targetNodes = queryResult.getNodes(retrieveId(relationshipTemplates[i].getTargetElement(), CMDBfMessages.relationshipTarget));
			
			if (sourceNodes == null || targetNodes == null)
			{
				continue;
			}
			IGraphElement[] sourceElements = sourceNodes.getElements();
			IGraphElement[] targetElements = targetNodes.getElements();
			
			List<IGraphElement> revisedSourceList = retrieveItems(items, relationshipTemplates[i].getSourceElement());
			List<IGraphElement> revisedTargetList = retrieveItems(items, relationshipTemplates[i].getTargetElement());
			
			// For every source item
			for (int j = 0; j < sourceElements.length; j++)
			{				
				IEdges currentEdges = null;
				boolean foundRelationship = false;
				
				// For every target item
				for (int k = 0; k < targetElements.length; k++)
				{
					// Does the relationship exist?
					currentEdges = retrieveRelationship(queryResult, relationshipTemplates[i], (IItem)sourceElements[j], (IItem)targetElements[k]);
					IGraphElement[] elements = currentEdges == null ? null : currentEdges.getElements();
					if (elements != null && elements.length > 0)
					{
						foundRelationship = true;
						
						if (!revisedTargetList.contains(targetElements[k]))
						{
							revisedTargetList.add(targetElements[k]);
						}
						// Add the relationships found
						for (int l = 0; l < elements.length; l++)
						{
							IRelationship relationship = (IRelationship) elements[l];
							edges.addRelationship(relationship);
						}
					}
				}
				
				// Add the source if a relationship between the j-th source 
				// and any of the target elements were found 
				if (foundRelationship && !revisedSourceList.contains(sourceElements[j]))
				{
					revisedSourceList.add(sourceElements[j]);										
				}

			}
		
			IGraphElement[] relationships = edges.getElements();
			if (relationships != null && relationships.length > 0)
			{
				queryResult.addEdges(edges);
			}
		}
		
		// Walk through each of the nodes that were touched
		for (Iterator<String> endpointIds = items.keySet().iterator(); endpointIds.hasNext();)
		{
			String endpointId = endpointIds.next();
			List<IGraphElement> itemList = items.get(endpointId);
			
			// If it's empty, then remove from result
			if (itemList.isEmpty())
			{
				queryResult.removeNodes(queryResult.getNodes(endpointId));
			}
			// Otherwise add elements that matched relationships found
			else
			{
				INodes nodes = queryResult.getNodes(endpointId);
				nodes.setElements(itemList.toArray(new IGraphElement[itemList.size()]));
			}			
		}		
	}
	
	private List<IGraphElement> retrieveItems(Map<String, List<IGraphElement>> items, IRelationshipEndpoint endpoint)
	{
		String endpointId = endpoint.getId();
		List<IGraphElement> list = items.get(endpointId);
		if (list == null)
		{
			list = new ArrayList<IGraphElement>();
			items.put(endpointId, list);
		}
		return list;
	}


	private String retrieveId(IItemReference itemReference, String source) throws CMDBfServiceException
	{
		if (itemReference == null || itemReference.getRef() == null)
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.SENDER, 
					CMDBfServiceException.UNKNOWN_TEMPLATE_ID, 
					NLS.bind(CMDBfMessages.faultsQueryUnknownTemplate, source), 
					CMDBfInternalUtility.appendNode(
							CMDBfInternalUtility.createElement(ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE, ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.GRAPH_ID), 
							CMDBfInternalUtility.createTextNode(itemReference.getId())));
		}
		return itemReference.getId();
	}


	private IEdges retrieveRelationship(IQueryResult context, IRelationshipTemplate relationshipTemplate, IItem source, IItem target) throws CMDBfServiceException
	{		
		IRecordConstraint[] recordConstraint = relationshipTemplate.getRecordConstraints();
		List<IConstraint[]> constraints = new ArrayList<IConstraint[]>();
		constraints.add(relationshipTemplate.getInstanceIdConstraint() == null ? null : new IConstraint[]{relationshipTemplate.getInstanceIdConstraint()});
		if (recordConstraint != null)
		{
			for (int i = 0; i < recordConstraint.length; i++)
			{
				constraints.add(recordConstraint[i].getRecordTypes());
				constraints.add(recordConstraint[i].getPropertyValues());
			}
		}
		IXPathExpression[] xpathExpressions = relationshipTemplate.getXpathExpressions();
		constraints.add(xpathExpressions);
		
		
		// Check to make sure that the intermediate template id of the depth limit is valid
		IDepthLimit depthLimit = relationshipTemplate.getDepthLimit();
		if (depthLimit != null)
		{
			retrieveId (depthLimit.getIntermediateItemTemplate(), CMDBfMessages.relationshipIntermediate);
		}
		
		// For each of the constraints
		IRelationshipTemplateHandler relationshipTemplateHandler = handlerFactory.createRelationshipTemplateHandler();
		if (!relationshipTemplateHandler.isInitialized())
		{
			relationshipTemplateHandler.initialize(getInit());
		}
		IEdges edges = relationshipTemplateHandler.execute(context, relationshipTemplate, source, target);
		for (int i = 0, constraintCount = constraints.size(); i < constraintCount; i++)
		{
			if (constraints.get(i) != null && constraints.get(i).length > 0)
			{
				edges = applyRelationshipConstraints(edges, constraints.get(i));
			}
		}
	
		return edges;
	}

	private IEdges applyRelationshipConstraints(IEdges parent, IConstraint[] constraints) throws CMDBfServiceException
	{
		if (constraints == null || constraints.length <= 0)
		{
			return null;
		}
		
		IRelationshipConstraintHandler constraintHandler = handlerFactory.createRelationshipConstraintHandler(constraints[0]);
		if (!constraintHandler.isInitialized())
		{
			constraintHandler.initialize(getInit());
		}
		boolean status = true;		
		IEdges result = parent;
		for (int i = 0; status && i < constraints.length; i++)
		{			
			result = constraintHandler.execute(result, constraints[i]);
			result.setStartingContext(false);
		}
				
		return result;
	}
	
	
	private INodes applyItemConstraints(INodes context, IConstraint[] constraints) throws CMDBfServiceException
	{		
		INodes result = context;		
		if (constraints == null || constraints.length <= 0)
		{
			return result;
		}
				
		IItemConstraintHandler constraintHandler = handlerFactory.createItemConstraintHandler(constraints[0]);
		if (!constraintHandler.isInitialized())
		{
			constraintHandler.initialize(getInit());
		}
		for (int i = 0; i < constraints.length; i++)
		{			
			result = constraintHandler.execute(result, constraints[i]);
			result.setStartingContext(false);
		}		
		return result;
	}

	
	private void processContentSelectors (IQueryResult result, IQuery query)
	{
		int itemTemplateCount = query.getItemTemplateCount();
		IItemTemplate[] itemTemplates = query.getItemTemplates();
		IRelationshipTemplate[] relationshipTemplates = query.getRelationshipTemplates();
				
		for (int i = 0, templateCount = itemTemplateCount + query.getRelationshipTemplateCount(); i < templateCount; i++)
		{
			boolean isItemTemplate = i < itemTemplateCount;
			processContentSelectors (
					result, 
					isItemTemplate ? itemTemplates[i] : relationshipTemplates[i - itemTemplateCount],
					isItemTemplate);
		}		
	}
	
	
	private void processContentSelectors(IQueryResult result, ITemplate template, boolean isItemTemplate)
	{
		String templateId = template.getId();

		// Exclude the template result if the suppress from result attribute is set
		if (template.isSuppressFromResult())
		{				
			if (isItemTemplate)
			{
				INodes nodes = result.getNodes(templateId);
				if (nodes != null)
				{
					result.removeNodes(nodes);
				}
			}
			else
			{
				IEdges edges = result.getEdges(templateId);
				if (edges != null)
				{
					result.removeEdges(edges);
				}
			}		
			
			return;
		}
		
		// Process the content selector of the template
		IContentSelector contentSelector = template.getContentSelector();			
		if (contentSelector == null)
		{
			return;
		}
		
		IQueryServiceElementCollection elementCollection = isItemTemplate ? result.getNodes(templateId) : result.getEdges(templateId);
		boolean matchedRecords = contentSelector.isMatchedRecords();
		
		
		IGraphElement[] graphElements = elementCollection.getElements();
		for (int j = 0; j < graphElements.length; j++)
		{				
			findMatch (contentSelector, graphElements[j], matchedRecords);
		}
	}


	private void findMatch(IContentSelector contentSelector, IGraphElement graphElement, boolean matchedRecords)
	{
		IRecord[] records = graphElement.getRecords();
		ISelectedRecordType[] selectedRecordTypes = contentSelector.getSelectedRecordTypes();
		
		if (selectedRecordTypes == null)
			return;
		
		// For every selected record type
		for (int i = 0; i < selectedRecordTypes.length; i++)
		{
			// For every record of the graph element
			for (int j = 0; j < records.length; j++)
			{								
				// Parse the value corresponding to the record in a DOM tree
				StringWriter sw = new StringWriter();
				records[j].getValue().toXML(sw, 0);			
				Node rootNode = null;
				try
				{
					rootNode = CMDBfInternalUtility.domParseDocument(new ByteArrayInputStream(sw.getBuffer().toString().getBytes())).getFirstChild();					
				} 
				catch (Exception e)
				{
					e.printStackTrace();
					continue;
				} 
				
				graphElement.removeRecord(records[j]);
				
				// Determine if there is a record match
				if (!selectedRecordTypes[i].getNamespace().toString().equals(rootNode.getNamespaceURI()) ||
					!selectedRecordTypes[i].getLocalName().equals(rootNode.getLocalName()))
				{
					continue;
				}
								
				// Process the properties							
				retrieveProperties(rootNode, selectedRecordTypes[i].getSelectedProperties());
				IRecord record = getQueryOutputArtifactFactory().createRecord(graphElement, null);
				record.setRecordMetadata(records[j].getRecordMetadata());
				record.setValue(new NodeXMLWritable(rootNode.hasChildNodes() ? rootNode : null));
				graphElement.addRecord(record);								
			}
		}
	}
		
		
	private void retrieveProperties(Node rootNode, IPropertyId[] properties)
	{				
		List<Node> candidateNodes = new ArrayList<Node>();
		findCandidateNodes (rootNode, candidateNodes, properties);
				
		// Remove the children of root
		if (rootNode.hasChildNodes())
		{
			NodeList children = rootNode.getChildNodes();
			for (int i = 0, childCount = children.getLength(); i < childCount; i++)
			{
				rootNode.removeChild(children.item(0));
			}
		}
		
		// Add the ones found
		for (int i = 0, candidateCount = candidateNodes.size(); i < candidateCount; i++)
		{
			rootNode.appendChild(candidateNodes.get(i));
		}
	}
	
	
	private void findCandidateNodes (Node node, List<Node> candidateNodes, IPropertyId[] properties)
	{
		if (!node.hasChildNodes())
		{
			return;
		}
		
		NodeList children = node.getChildNodes();
		
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
						
			if (child.getNodeType() != Node.ELEMENT_NODE)
			{				
				continue;
			}
			
			for (int j = 0; j < properties.length; j++)
			{
				String namespace = properties[j].getNamespace().toString();
				String localName = properties[j].getLocalName();
				boolean added = false;
				
				// Add to candidate list if there is a name space and local name match
				if (namespace.equals(child.getNamespaceURI()) &&
					localName.equals(child.getLocalName()))
				{
					candidateNodes.add(child);
					added = true;
				}
				// Or there is an attribute match
				else
				{
					NamedNodeMap atts = child.getAttributes();
					for (int k = 0, attCount = atts.getLength(); k < attCount; k++)
					{
						Node currentAtt = atts.item(k);
						if ((namespace.equals(currentAtt.getNamespaceURI()) || namespace.equals(child.getNamespaceURI())) &&
							localName.equals(currentAtt.getLocalName()))
						{
							candidateNodes.add(child);
							added = true;
							break;
						}
					}
				}
				
				if (!added)
				{
					findCandidateNodes (child, candidateNodes, properties);
				}
			}			
		}	
	}


	protected IQueryOutputArtifactFactory getQueryOutputArtifactFactory() {
		if (queryOutputArtifactFactory == null) {
			queryOutputArtifactFactory = QueryOutputArtifactFactory.getInstance();
		}
		return queryOutputArtifactFactory;
	}
}


	
