/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.registration.client;

import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.Constants;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.apache.axis2.util.XMLUtils;
import org.eclipse.cosmos.dc.provisional.cmdbf.client.CMDBfServiceClient;
import org.eclipse.cosmos.dc.provisional.cmdbf.exception.CMDBfException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.transform.DeregistrationOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterRequest;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.RegistrationOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.input.artifacts.IRegisterRequest;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.TransformerException;
import org.w3c.dom.Element;

/**
 * Remote client for a CMDBf registration service.
 *
 */
public class RegistrationServiceClient extends CMDBfServiceClient {


	/**
	 * Creates a registration service client, given the EPR of the service
	 * @param epr Endpoint reference URL of the registration service
	 */
	public RegistrationServiceClient(String epr) {
		super( epr );
	}
	
	/**
	 * Creates a registration service client, given the EnpointReference object 
	 * corresponding to the registration service.
	 * @param epr EndpointReference object representing the registration service EPR
	 */
	public RegistrationServiceClient(EndpointReference epr) {
		super( epr );
	}

	/**
	 * Registration operation of the CMDBf registration service, where input and output 
	 * are represented in document object model (DOM). 
	 * 
	 * @param registrationRequest CMDBf registration request, represented in DOM. 
	 * @return An element representing the CMDBf registration response, represented in DOM.
	 * @throws CMDBfException If error occurred during registration
	 */
	public Element register(Element registrationRequest) throws CMDBfException 
	{
		try {
			OMElement request = XMLUtils.toOM(registrationRequest);
			OMElement result = register(request);
			return XMLUtils.toDOM(result);
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * Registration operation of the CMDBf registration service, where input and output 
	 * are represented in COSMOS object model.  
	 * 
	 * @param registrationRequest CMDBf registration request
	 * @return CMDBf registration response
	 * @throws CMDBfException If error occurred during registration 
	 */
	public IRegisterResponse register(IRegisterRequest registrationRequest) throws CMDBfException {
		try {
			StringWriter writer = new StringWriter();
			registrationRequest.toXML(writer, 0);
			StringReader regReader = new StringReader(writer.toString());
			
			//create the parser
			XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(regReader);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
	
			//get the root element
			OMElement requestPayload = builder.getDocumentElement();
			
			OMElement result = register(requestPayload);

	        IRegisterResponse resultObj = RegistrationOutputTransformer.transform(new StringReader(result.toString()));
	        
	        return resultObj;
		} catch (XMLStreamException e) {
			throw new CMDBfException(e);
		} catch (TransformerException te) {
			throw new CMDBfException(te);
		}
	}
	
	/**
	 * Registration operation of the CMDBf registration service, where input and output are
	 * represented in AXIOM object model. 
	 * 
	 * @param registrationRequest CMDBf registration request, in AXIOM object representation
	 * @return CMDBf registration response, in AXIOM object representation
	 * @throws CMDBfException If error occurred during registration
	 */
	public OMElement register(OMElement registrationRequest) throws CMDBfException {
		try {
	        Options options = new Options();
	        options.setTo(serviceEPR);
	        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
	        
	        if ( soapVersion == 11 )
		        options.setSoapVersionURI(org.apache.axiom.soap.SOAP11Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	        else
	        	options.setSoapVersionURI(org.apache.axiom.soap.SOAP12Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	
	        ServiceClient sender = new ServiceClient();
	        
	        //
	        // Basic security?
	        //
	        if ( username != null && password != null ){
	        	//
	        	// construct security header and add to SOAP header...
	        	//
	        	addBasicSecurityHeader( sender, username, password);
	        }
	        
	        sender.setOptions(options);
	        OMElement result = sender.sendReceive(registrationRequest);
	        return result;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * Deregistration operation of the CMDBf registration service, where input and output 
	 * are represented in document object model (DOM). 
	 * 
	 * @param deregistrationRequest The deregistration request, representated in a DOM Element object 
	 * @return An element representing the CMDBf deregistration response, representated in a DOM Element object
	 * @throws CMDBfException If error occurred during deregistration
	 */
	public Element deregister(Element deregistrationRequest) throws CMDBfException 
	{
		try {
			OMElement request = XMLUtils.toOM(deregistrationRequest);
			OMElement result = deregister(request);
			return XMLUtils.toDOM(result);
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * Deregistration operation of the CMDBf registration service, where input and output 
	 * are represented in COSMOS object model. 
	 * 
	 * @param deregistrationRequest The deregistration request
	 * @return An element representing the CMDBf deregistration response
	 * @throws CMDBfException If error occurred during deregistration
	 */
	public IDeregisterResponse deregister(IDeregisterRequest deregistrationRequest) throws CMDBfException {
		try {
			StringWriter writer = new StringWriter();
			deregistrationRequest.toXML(writer, 0);
			StringReader reader = new StringReader(writer.toString());
			
			//create the parser
			XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(reader);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
	
			//get the root element
			OMElement requestPayload = builder.getDocumentElement();
			
			OMElement result = deregister(requestPayload);

	        IDeregisterResponse resultObj = DeregistrationOutputTransformer.transform(new StringReader(result.toString()));
	        
	        return resultObj;
		} catch (XMLStreamException e) {
			throw new CMDBfException(e);
		} catch (TransformerException te) {
			throw new CMDBfException(te);
		}

	}
	
	/**
	 * Deregistration operation of the CMDBf registration service, where input and output 
	 * are represented in AXIOM object model. 
	 * 
	 * @param deregistrationRequest deregistrationRequest The deregistration request
	 * @return An element representing the CMDBf deregistration response
	 * @throws CMDBfException If error occurred during deregistration
	 */
	public OMElement deregister(OMElement deregistrationRequest) throws CMDBfException {
		try {
	        Options options = new Options();
	        options.setTo(serviceEPR);
	        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
	        
	        if ( soapVersion == 11 )
		        options.setSoapVersionURI(org.apache.axiom.soap.SOAP11Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	        else
	        	options.setSoapVersionURI(org.apache.axiom.soap.SOAP12Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	
	        ServiceClient sender = new ServiceClient();
	        
	        //
	        // Basic security?
	        //
	        if ( username != null && password != null ){
	        	//
	        	// construct security header and add to SOAP header...
	        	//
	        	addBasicSecurityHeader( sender, username, password);
	        }
	        
	        sender.setOptions(options);
	        OMElement result = sender.sendReceive(deregistrationRequest);
	        return result;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}

}
