/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.internal.cmdbf.services.query.transform.input.artifacts;

import java.io.StringWriter;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRelationshipEndpoint;

/**
 * Represents <source> and <target> elements in relationship template
 * 
 * @author David Whiteman
 */
public class RelationshipEndpoint extends ItemReference implements IRelationshipEndpoint 
{
	
	private int minimum; // optional cardinality, initialized to NO_CARDINALITY
	private int maximum; // optional cardinality, initialized to NO_CARDINALITY
	private int type;
	
	/**
	 * Create an endpoint on a relationship.  The type argument
	 * indicates whether the endpoint is the SOURCE or TARGET of
	 * the relationship.
	 * 
	 * @param type
	 */
	public RelationshipEndpoint(int type) {
		super();
		this.type = type;
		this.minimum = NO_CARDINALITY;
		this.maximum = NO_CARDINALITY;
	}


	/**
	 * Answer the minimum cardinality on the relationship.
	 * A response of NO_CARDINALITY indicates that no minimum
	 * was set.
	 * 
	 * @return int minimum cardinality
	 */
	public int getMinimum() {
		return minimum;
	}

	public void setMinimum(int minimum) {
		this.minimum = minimum;
	}

	/**
	 * Answer the maximum cardinality on the relationship.
	 * A response of NO_CARDINALITY indicates that no maximum
	 * was set.
	 * 
	 * @return int maximum cardinality
	 */
	public int getMaximum() {
		return maximum;
	}

	public void setMaximum(int maximum) {
		this.maximum = maximum;
	}

	/**
	 * Answer whether the referenced item template
	 * represents the source or the target in the relationship.
	 * 
	 * @return int (SOURCE or TARGET)
	 */
	public int getType() {
		return type;
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		String relationshipType = ICMDBfServicesConstants.CMDBF_PREFIX + (getType() == SOURCE ? IQueryTransformerConstants.SOURCE_TEMPLATE_ELEMENT : IQueryTransformerConstants.TARGET_TEMPLATE_ELEMENT);
		writer.write(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET+relationshipType+ICMDBfServicesConstants.SINGLE_SPACE);
		CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.REF_ATTRIBUTE, getRef().getId());

		if (getMinimum() != NO_CARDINALITY) {
			writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.MINIMUM_ATTRIBUTE, Integer.toString(getMinimum()));
		}
		if (getMaximum() != NO_CARDINALITY) {
			writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.MAXIMUM_ATTRIBUTE, Integer.toString(getMaximum()));
		}

		writer.write(ICMDBfServicesConstants.FORWARD_SLASH + ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET + ICMDBfServicesConstants.nl);
	}

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + maximum;
		result = prime * result + minimum;
		result = prime * result + ((getRef() == null) ? 0 : getRef().hashCode());
		result = prime * result + type;
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final RelationshipEndpoint other = (RelationshipEndpoint) obj;
		if (maximum != other.maximum)
			return false;
		if (minimum != other.minimum)
			return false;
		if (getRef() == null) {
			if (other.getRef() != null)
				return false;
		} else if (!getRef().equals(other.getRef()))
			return false;
		if (type != other.type)
			return false;
		return true;
	}
}
