/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform;


import java.io.StringWriter;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.internal.cmdbf.services.transform.artifacts.CommonArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.IRootElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IEdges;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IAdditionalRecordType;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecordMetadata;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRelationship;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX parser that builds QueryResult object from XML input containing a <queryResult> element
 * 
 * 
 * @author David Whiteman
 */
public class QueryOutputStreamHandler extends DefaultHandler implements ITransformerHandler 
{
	private IQueryResult queryResult;
	private IGraphElement currentResultEntity;
	private String currentElementData;
	private INodes currentNodes;
	private IEdges currentEdges;
	private IRecord currentRecord;
	private IInstanceId currentInstanceId;
	private IRecordMetadata currentRecordMetadata;
	private IQueryOutputArtifactFactory artifactFactory;
	
	public QueryOutputStreamHandler(IQueryOutputArtifactFactory artifactFactory) {
		this.artifactFactory = artifactFactory;
	}

	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException 
	{
		if (ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			if (IQueryTransformerConstants.QUERY_RESULT_ELEMENT.equals(localName)) 
			{
				queryResult = getArtifactFactory().createQueryResult();
			} 
			
			else if (ICMDBfServicesConstants.ITEM_ELEMENT.equals(localName)) 
			{
				currentResultEntity = getArtifactFactory().createItem();
				currentNodes.addItem((IItem) currentResultEntity);
			} 
			
			else if (ICMDBfServicesConstants.RELATIONSHIP_ELEMENT.equals(localName)) 
			{
				currentResultEntity = getArtifactFactory().createRelationship();
				currentEdges.addRelationship((IRelationship) currentResultEntity);
			} 
			
			else if (ICMDBfServicesConstants.SOURCE_ELEMENT.equals(localName)) 
			{
				currentInstanceId = CommonArtifactFactory.getInstance().createInstanceId();
			} 
			
			else if (ICMDBfServicesConstants.TARGET_ELEMENT.equals(localName)) 
			{
				currentInstanceId = CommonArtifactFactory.getInstance().createInstanceId();
			} 
			
			else if (IQueryTransformerConstants.NODES_ELEMENT.equals(localName)) 
			{
				String templateId = attributes.getValue(IQueryTransformerConstants.TEMPLATE_ID_ATTRIBUTE);
				currentNodes = getArtifactFactory().createNodes(templateId);
				queryResult.addNodes(currentNodes);
			} 
			
			else if (IQueryTransformerConstants.EDGES_ELEMENT.equals(localName)) 
			{
				String templateId = attributes.getValue(IQueryTransformerConstants.TEMPLATE_ID_ATTRIBUTE);
				currentEdges = getArtifactFactory().createEdges(templateId);
				queryResult.addEdges(currentEdges);
			} 
			
			else if (ICMDBfServicesConstants.RECORD_ELEMENT.equals(localName)) 
			{
				currentRecord = CommonArtifactFactory.getInstance().createRecord(currentResultEntity, null);
				currentResultEntity.addRecord(currentRecord);
				currentElementData = "";
				
				CMDBfInternalUtility.addNamespaces(currentRecord, attributes);			
			}
			
			else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
			{
				currentInstanceId = CommonArtifactFactory.getInstance().createInstanceId();
			} 
			
			else if (ICMDBfServicesConstants.ADDITIONAL_RECORD_TYPE_ELEMENT.equals(localName)) 
			{
				String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
				String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
				IAdditionalRecordType adt = CommonArtifactFactory.getInstance().createAdditionalRecordType(namespace, locName);
				currentResultEntity.addAdditionalRecordType(adt);
			} 
			
			// <recordMetadata>
			else if (currentRecord != null && ICMDBfServicesConstants.RECORD_METADATA_ELEMENT.equals(localName)) 
			{
				currentRecord.setValueFromString(currentElementData.trim());
				currentRecordMetadata = CommonArtifactFactory.getInstance().createRecordMetadata();		
				currentRecord.setRecordMetadata(currentRecordMetadata);
				currentElementData = null;
			}
		}
		else if (currentRecord != null && currentRecordMetadata == null) 
		{
			StringWriter tempWriter = new StringWriter();
			CMDBfServicesUtil.addIndent(tempWriter, 4);
			tempWriter.write(CMDBfServicesUtil.beginTagFor(name, attributes));
			tempWriter.write(ICMDBfServicesConstants.nl);
			currentElementData += tempWriter.toString();
			if (!name.equals(localName)) 
			{
				String prefix = name.split(ICMDBfServicesConstants.COLON)[0];
				currentRecord.addNamespace(prefix, uri);
			}
		} 
	}

	protected IQueryOutputArtifactFactory getArtifactFactory() {
		if (artifactFactory == null) {
			artifactFactory = QueryOutputArtifactFactory.getInstance();
		}
		return artifactFactory;
	}

	public void endElement(String uri, String localName, String name) throws SAXException 
	{
		if (ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			if (ICMDBfServicesConstants.MDR_ID_ELEMENT.equals(localName)) 
			{
				currentInstanceId.setMdrId(currentElementData.trim());
				currentElementData = null;
			} 
			
			else if (ICMDBfServicesConstants.LOCAL_ID_ELEMENT.equals(localName)) 
			{
				currentInstanceId.setLocalId(currentElementData.trim());
				currentElementData = null;
			} 
			
			else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
			{
				currentResultEntity.addInstanceId(currentInstanceId);
				currentInstanceId = null;
			} 
			
			else if (ICMDBfServicesConstants.SOURCE_ELEMENT.equals(localName)) 
			{
				((IRelationship) currentResultEntity).setSourceId(currentInstanceId);
				currentInstanceId = null;
			} 
			
			else if (ICMDBfServicesConstants.TARGET_ELEMENT.equals(localName)) 
			{
				((IRelationship) currentResultEntity).setTargetId(currentInstanceId);
				currentInstanceId = null;
			} 
			
			else if (ICMDBfServicesConstants.RECORD_ELEMENT.equals(localName)) 
			{
				currentRecord = null;
			} 
			
			// <recordMetadata>
			else if (ICMDBfServicesConstants.RECORD_METADATA_ELEMENT.equals(localName))
			{
				currentRecordMetadata = null;
			}
			
			// <recordId>
			else if (currentRecordMetadata != null && ICMDBfServicesConstants.RECORD_ID_ELEMENT.equals(localName)) 
			{
				currentRecordMetadata.setRecordId(CMDBfServicesUtil.createURI(currentElementData.trim()));
				currentElementData = null;
			}

			// <lastModified>
			else if (currentRecordMetadata != null && ICMDBfServicesConstants.LAST_MODIFIED_ELEMENT.equals(localName)) 
			{
				currentRecordMetadata.setLastModified(currentElementData.trim());
				currentElementData = null;
			}
			
			// <baselineId>
			else if (currentRecordMetadata != null && ICMDBfServicesConstants.BASE_LINE_ID_ELEMENT.equals(localName)) 
			{
				currentRecordMetadata.setBaselineId(currentElementData.trim());
				currentElementData = null;
			}

			// <snapshotId>
			else if (currentRecordMetadata != null && ICMDBfServicesConstants.SNAPSHOT_ID_ELEMENT.equals(localName)) 
			{
				currentRecordMetadata.setSnapshotId(currentElementData.trim());
				currentElementData = null;
			}
		}		
		else if (currentRecord != null && currentRecordMetadata == null) 
		{
			StringWriter tempWriter = new StringWriter();
			CMDBfServicesUtil.addIndent(tempWriter, 4);
			tempWriter.write(CMDBfServicesUtil.endTagFor(name));
			tempWriter.write(ICMDBfServicesConstants.nl);
			currentElementData += tempWriter.toString();
		}
	}

	public void characters(char[] ch, int start, int length)
			throws SAXException {
		String tempData = new String(ch, start, length).trim();
		if (currentElementData == null) {
			currentElementData = tempData;
		} else {
			StringWriter tempWriter = new StringWriter();
			if (currentRecord != null) {
				CMDBfServicesUtil.addIndent(tempWriter, 4);
			}
			tempWriter.write(tempData);
			if (currentRecord != null) {
				tempWriter.write('\n');
			}
			currentElementData += tempWriter.toString();
		}
	}

	public IQueryResult getQueryResult() {
		return queryResult;
	}

	public void error(SAXParseException e) throws SAXException {
		throw e;
	}

	public IRootElement getResult() {
		return (IRootElement) getQueryResult();
	}


}
