/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeNode;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * Used to build an inheritance map that will indicate the relationships between
 * the reference types.
 * 
 * This is a rewrite of the old TypeInheritanceDataBuilderImpl class, using PSVI
 * to get type data instead of parsing the schema directly.  Although it uses PSVI,
 * this class is a candidate for deletion, as this kind of data can be dynamically
 * obtained in the other builders without re-parsing.
 * 
 * @deprecated
 * @author sleeloy
 * @author Ali Mehregani
 * @author David Whiteman
 */
public class TypeInheritanceDataBuilder extends AbstractDataBuilder<Map<String, Map<String, TypeNode>>>
{
	
	/**
	 * The static ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".TypeInheritanceDataBuilder";

	/**
	 * The inheritance map
	 * 
	 * KEY = An object of type {@link String} indicating the uri
	 * VALUE = A map, where
	 * 	KEY = The local name of the complex type
	 * 	VALUE = An object of type {@link TypeNode} indicating the immediate parent of KEY 
	 */
	private Map<String, Map<String, TypeNode>> inheritanceMap = new HashMap<String, Map<String, TypeNode>>();

	/**
	 * Stores the attributes of a complex type
	 * 
	 * KEY = An object of type {@link String} indicating the uri
	 * VALUE = A map, where
	 * 	KEY = The complex type name
	 * 	VALUE = An object of type {@link Map} representing the attributes of KEY
	 */
	private Map<String, Map<String, Map<String, String>>> attributes;
	
	/**
	 * Constructor
	 */
	public TypeInheritanceDataBuilder()
	{
		attributes = new Hashtable<String, Map<String, Map<String, String>>>();
	}
	
	/**
	 * The data structure is a {@link Map}, where 
	 * 
	 * KEY = An object of type {@link String} indicating a reference
	 * VALUE = An object of type {@link String} indicating the immediate parent of KEY
	 *  
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public Map<String, Map<String, TypeNode>> getDataStructure()
	{
		return inheritanceMap;
	}

	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	@SuppressWarnings("unchecked")
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		//super.startElement(uri, localName, qName, attributes);
		if (ISMLConstants.SCHEMA_URI.equals(uri)) {
			return;
		}
		ElementPSVI elementPsvi = getPsvi().getElementPSVI();
		if (elementPsvi == null) {
			return;
		}
		XSTypeDefinition typeDef = elementPsvi.getTypeDefinition();
		if ((typeDef == null) || (typeDef.getTypeCategory() != XSTypeDefinition.COMPLEX_TYPE) || (typeDef.getNamespace().equals(ISMLConstants.SCHEMA_URI))) {
			return;
		}
		String complexTypeName = typeDef.getName();
		if (complexTypeName == null) {
			return;
		}
		Hashtable<String, String> complexTypeAtts = new Hashtable<String, String>();
		AnnotationResult annotationResult = SMLValidatorUtil.retrieveAnnotation(typeDef, ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE, true);
		if (annotationResult != null) {
			complexTypeAtts.put(ISMLConstants.ACYCLIC_ATTRIBUTE, annotationResult.getNodes()[0].getNodeValue());
		}
				
		String targetNameSpace = typeDef.getNamespace();						
				
		TypeNode currentComplexType = new TypeNode(complexTypeName, (targetNameSpace == null ? IValidationConstants.EMPTY_STRING : targetNameSpace));
		Map typesPerUri = SMLValidatorUtil.retrieveNestedMap(this.attributes, currentComplexType.getUri(), true);
		typesPerUri.put(currentComplexType.getType(), complexTypeAtts);

		int derivationMethod = ((XSComplexTypeDefinition)typeDef).getDerivationMethod();
		if ((XSConstants.DERIVATION_EXTENSION == derivationMethod) || 
				(XSConstants.DERIVATION_RESTRICTION == derivationMethod)) {
			if (typeDef.getBaseType().getNamespace().equals(ISMLConstants.SCHEMA_URI)) {
				return;
			}
			String parentTypeName = typeDef.getBaseType().getName();
			String parentTypeNamespace = typeDef.getBaseType().getNamespace();
			
			/* Before adding the child-parent relation, we need to make sure that the relationship
			 * is valid (e.g. a child type that has acyclic set to false cannot extend a parent 
			 * with acyclic set to true) -- making the assumption that parent types are declared
			 * before child types */
			TypeNode parent = new TypeNode(parentTypeName, parentTypeNamespace);
			validateRelationship(parent, currentComplexType);
			Map namesPerUri = SMLValidatorUtil.retrieveNestedMap(inheritanceMap, currentComplexType.getUri(), true);
			namesPerUri.put(currentComplexType.getType(), parent);
		}
	}
	


	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */	
	public void endElement(String uri, String localName, String qName) throws SAXException {
	}


	/**
	 * Based on the inheritance map passed in, this method will return true if and only if
	 * refTypeInQuestion is a derived type of refType.
	 * 
	 * @param inheritanceMap The inheritance map
	 * @param possibleAncestorType A primary ref type 
	 * @param refType The reference type in question
	 * @return true iff refType is a derived type of possibleAncestorType; false otherwise
	 */
	public static boolean isDerivedType(Map<String, Map<String, TypeNode>> inheritanceMap, TypeNode possibleAncestorType, TypeNode refType)
	{
		/* A type is always a derived type of itself */
		if (possibleAncestorType.equals(refType)) {
			return true;
		}
		
		TypeNode currentType = refType;
		while (currentType != null) {
			Map<String, TypeNode> namePerUri = inheritanceMap.get(currentType.getUri());
			currentType = namePerUri == null ? null : (TypeNode) namePerUri.get(currentType.getType());
			if (possibleAncestorType.equals(currentType)) {
				return true;
			}
		}
		
		return false;
	}

	
	/**
	 * Validates the relationship between a parent and a child type
	 * 
	 * @param baseType The parent type
	 * @param complexType The child type
	 */
	private void validateRelationship(TypeNode baseType, TypeNode complexType) {
		if (isTypeAcyclic(baseType, false) && !isTypeAcyclic(complexType, true)) {
			setStructureValidity(false);
			if (getErrorMessage() == null) {
				setErrorMessage(ValidationMessageFactory.createErrorMessage(getLocator().getLineNumber(), NLS.bind(SMLValidationMessages.acyclicBadDerivation, new String[] { baseType.getType(), complexType.getType() })));
			} else {
				appendToErrorMessage(NLS.bind(SMLValidationMessages.acyclicBadDerivation, new String[] { baseType.getType(), complexType.getType() }));
			}
		}
	}
	
	
	/**
	 * Indicates if a specific type is acyclic
	 * 
	 * @param complexType The type
	 * @param defaultValue The default value that will be re
	 * @return true if 'type' is acyclic; false otherwise
	 */
	private boolean isTypeAcyclic(TypeNode complexType, boolean defaultValue) {
		if (complexType.getUri() == null || complexType.getType() == null) {			
			return false;
		}
		
		Map<String, Map<String, String>> namesPerUri = attributes.get(complexType.getUri());
		Map<String, String> typeAttributes = namesPerUri == null ? null : namesPerUri.get(complexType.getType());
		if (typeAttributes != null) {
			String acyclicValue = (String)typeAttributes.get(ISMLConstants.ACYCLIC_ATTRIBUTE);
			if (acyclicValue != null) {
				return IValidationConstants.TRUE_VALUE.equalsIgnoreCase(acyclicValue);
			}
		}
		
		return defaultValue;
	}
	
	
	public static TypeNode retrieveType(Map<String, Map<String, TypeNode>> inheritance, String uri, String localName) {
		Map<String, TypeNode> typesPerUri = inheritance.get(uri);
		return typesPerUri == null ? null : typesPerUri.get(localName);
	}
}
