/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.reference;

import java.net.URI;

import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.reference.XPointer;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.w3c.dom.Node;

/**
 * The fully qualified SML reference includes the document reference 
 * and the XPointer reference.  
 * 
 * @author Ali Mehregani
 * @author John Arwe
 * @author David Whiteman
 */
public class SMLQualifiedReference
{
	private static final String LOCALHOST = "localhost";
	private static final String PROTOCOL_FILE = "file";

	/**
	 * The repository
	 */
	private ISMLRepository repository;
	
	/**
	 * The document reference
	 */
	private String documentReference;

	/**
	 * The xpointer expression
	 */
	private String xpointer;

	/**
	 * The constructor 
	 * 
	 * @param reference The reference
	 */
	public SMLQualifiedReference(ISMLRepository repository, String reference) {
		this.repository = repository;
		int anchorInx = reference.indexOf('#');
		documentReference = anchorInx >= 0 ? reference.substring(0, anchorInx) : reference;
		xpointer = anchorInx >= 0 ? reference.substring(anchorInx + 1) : null;	
	}

	
	public Node resolveReference() {		
		try	{
			SMLDocumentReference docReference = new SMLDocumentReference(repository, documentReference);
			String contextDirectory = repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, ISMLConstants.EMPTY_STRING);
			URI baseURI;
			if (!contextDirectory.equals(ISMLConstants.EMPTY_STRING)) {	//	local filepath exists, make it into a URI
				if (!contextDirectory.startsWith(ISMLConstants.FORWARD_SLASH)) {	//	if does not (yet) start with /, do so in order to make URI construction work as expected
					contextDirectory = ISMLConstants.FORWARD_SLASH + contextDirectory;
				}
				if (!contextDirectory.endsWith(ISMLConstants.FORWARD_SLASH)) {	//	if does not (yet) end with /, do so in order to make relative refs like file.xml work as expected
					//	If the path does not end with /, then file.xml would replace the last path segment instead of being appended to it.
					contextDirectory += ISMLConstants.FORWARD_SLASH;
				}
				//	Note that the constructor handles issues like escaping otherwise illegal path characters
				baseURI = new URI(PROTOCOL_FILE, LOCALHOST, contextDirectory, null, null);
			}
			else	//	context directory == "" , base code
			{
				baseURI = new URI(contextDirectory);
			}
			docReference.setBase(baseURI);
			Node documentNode = docReference.retrieveDocumentDOM();
			if (xpointer == null || documentNode == null) {
				return documentNode;
			}
			
			return (Node)XPointer.compile(xpointer).evaluate(documentNode.getFirstChild());
		} 
		catch (Exception e) {
			e.printStackTrace();
		} 
		
		return null;
	}


	/**
	 * @return the documentReference
	 */
	public String getDocumentReference() {
		return documentReference;
	}

	/**
	 * @param documentReference the documentReference to set
	 */
	public void setDocumentReference(String documentReference) {
		this.documentReference = documentReference;
	}

	/**
	 * @return the xpointer
	 */
	public String getXpointer() {
		return xpointer;
	}

	/**
	 * @param xpointer the xpointer to set
	 */
	public void setXpointer(String xpointer) {
		this.xpointer = xpointer;
	}		
}
