/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.common.validation;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collection;

import org.eclipse.cosmos.me.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;

/**
 * 
 * @author Eric S. Rose (esrose@us.ibm.com)
 *
 */
public interface SDDManager {
	// Descriptor type constants
	public static final int DEPLOYMENT_DESCRIPTOR = 1;
	public static final int PACKAGE_DESCRIPTOR = 2;
	
	/**
	 * Add a Document representing a descriptor to either the package descriptor or deployment descriptor list.
	 * Any ValidationRules already associated with the SDDManager will be added to this Descriptor.
	 * 
	 * @param descriptorDocument Document representing the descriptor to add
	 * @param descriptorFile XML file to associate with the Document
	 * @throws IllegalArgumentException if descriptorDocument or descriptorFile is null
	 */
	void addDescriptorFromDocument(Document descriptorDocument, File descriptorFile) throws IllegalArgumentException;
	
	/**
	 * Load a Document representing a descriptor from an XML file and add it to the Package or Deployment Descriptor list.
	 * Any ValidationRules already associated with the SDDManager will be added to this Descriptor.
	 * 
	 * @param descriptorFile XML file to associate with the Document
	 * @return The Document loaded from the XML file
	 * @throws IllegalArgumentException if descriptorFile is null
	 * @throws FileNotFoundException if the given File is missing
	 * @throws IOException if there's a problem reading from the given file
	 * @throws XMLValidationException if the given XML file isn't valid against the SDD schema
	 */
	Document addDescriptorFromFile(File descriptorFile)
			throws IllegalArgumentException, FileNotFoundException, IOException, XMLValidationException;
	
	/**
	 * Remove a Document from the list of descriptors
	 * 
	 * @param descriptorDocument Document to remove from the descriptor list
	 */
	void removeDescriptor(Document descriptorDocument);
	
	/**
	 * Write all Descriptors out to the associated XML files
	 * 
	 * @param performValidation Whether to perform XML validation
	 * @throws XMLValidationException if an errors occurs validating the XML
	 * @throws IOException if an errors occurs writing the XML to the file
	 */
	void writeAllDescriptors(boolean performValidation) throws XMLValidationException, IOException;

	/**
	 * Write the given Descriptors out to the associated XML files
	 * 
	 * @param descriptors Collection of descriptor Documents to write out
	 * @param performValidation Whether to perform validation before writing XML
	 * @throws XMLValidationException if an errors occurs validating the XML
	 * @throws IOException if an error occurs writing the XML to the file
	 */
	void writeDescriptors(Collection<Document> descriptors, boolean performValidation)
			throws XMLValidationException, IOException;
	
	/**
	 * Validate all descriptors against their Validation Rules
	 * 
	 * @return A Collection of XMLValidationErrors, or an empty list if no error occurred 
	 */
	Collection<XMLValidationError> validateAllDescriptors() throws XMLValidationException;
	
	/**
	 * Validate the given descriptors against their Validation Rules
	 * 
	 * @param descriptors A Collection of Descriptors to validate
	 * @return A Collection array of XMLValidationErrors, or an empty list if no error occurred
	 */
	Collection<XMLValidationError> validateDescriptors(Collection<Document> descriptors);
	
	/**
	 * Add a ValidationRule to the validation of the given descriptor type
	 * NOTE: If adding a SchemaValidationRule, it must NOT contain the SDD Schema files. The DAS will add them
	 * automatically and an IllegalArgumentException will result if they get added twice.
	 * 
	 * @param validationRule ValidationRule to add
	 * @param descriptorType Descriptor type (package or deployment) with which to associate the ValidationRule
	 * 
	 * @throws IllegalArgumentException if validationRule is null or descriptorType isn't PACKAGE or DEPLOYMENT
	 */
	void addValidation(ValidationRule validationRule, int descriptorType) throws IllegalArgumentException;
	
	/**
	 * Remove a ValidationRule from the validation of the given descriptor type
	 * 
	 * @param validationRule ValidationRule to remove
	 * @param descriptorType Descriptor type (package or deployment) from which to remove the ValidationRule
	 * 
	 * @throws IllegalArgumentException if descriptorType isn't PACKAGE or DEPLOYMENT
	 */
	void removeValidation(ValidationRule validationRule, int descriptorType) throws IllegalArgumentException;	
	
	/**
	 * Get the Package Descriptors associated with the SDDManager
	 * 
	 * @return A Collection of Documents containing the Package Descriptors
	 */
	Collection<Document> getPackageDescriptors();
	
	/**
	 * Get the Deployment Descriptors associated with the SDDManager
	 * 
	 * @return A Collection of Documents containing the Deployment Descriptors
	 */
	Collection<Document> getDeploymentDescriptors();

	/**
	 * Get all the Descriptors associated with the SDDManager
	 * 
	 * @return A Collection of Documents containing all the Descriptors
	 */
	Collection<Document> getAllDescriptors();
	
	/**
	 * Get the Package Descriptor ValidationRules
	 * 
	 * @return A Collection of ValidationRules associated with Package Descriptors
	 */
	Collection<ValidationRule> getPackageDescriptorRules();
	
	/**
	 * Get the Deployment Descriptor ValidationRules
	 * 
	 * @return A Collection of ValidationRules associated with Deployment Descriptors
	 */
	Collection<ValidationRule> getDeploymentDescriptorRules();

	/**
	 * Get the all ValidationRules associated with the SDDManager
	 * 
	 * @return A Collection of ValidationRules associated with the SDDManager
	 */
	Collection<ValidationRule> getAllValidationRules();
	
	/**
	 * Set the encoding to use when writing out XML documents. Set to null to use the default encoding.
	 * The default encoding will be used if this method is not called before writing out the XML. 
	 * 
	 * @param encoding Encoding to use when writing XML documents
	 */
	void setEncoding(String encoding);		
}