/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.common.validation.plugin;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Vector;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.eclipse.cosmos.me.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.sdd.resources.common.validation.Messages;

/**
 * 
 * @author Eric S. Rose (esrose@us.ibm.com) and Alan Watkins
 *
 */
public class SchematronValidationRule implements ValidationRule {
	
	private File schematronRulesFile;
		
	public SchematronValidationRule(File schematronRulesFile) throws IOException, FileNotFoundException {
		
		this.schematronRulesFile = schematronRulesFile;
		
		// Make sure the rules file exists
		if (!schematronRulesFile.exists()) {
			throw new FileNotFoundException(Messages.getString("SchematronValidationRule.0", schematronRulesFile.getName())); //$NON-NLS-1$
		}
	
		// Make sure the rules file can be read
		if (!schematronRulesFile.canRead()) {
			throw new IOException(Messages.getString("SchematronValidationRule.2", schematronRulesFile.getName())); //$NON-NLS-1$
		}
	}

	public Collection<XMLValidationError> validate(InputStream xmlStream) {
		StreamResult errorReportResult;
		Collection<XMLValidationError> validationErrorList = new Vector<XMLValidationError>();
		
		try {
			// Get an InputStream from the Schematron file and build a Transformer to compile the Schematron rules
			InputStream schematronInputStream = new FileInputStream(ValidatorUtils.getTempFileFromSchematronText());
			Transformer compileSchematronTransformer =
				TransformerFactory.newInstance().newTransformer(new StreamSource(schematronInputStream));

			// Get an InputStream from the Schematron rules file and compile the Schematron rules into an XSL
			InputStream schematronRulesInputStream = new FileInputStream(schematronRulesFile);
			StreamResult schematronXslResult = new StreamResult(new ByteArrayOutputStream());
			compileSchematronTransformer.transform(new StreamSource(schematronRulesInputStream),
					schematronXslResult);

			// Apply the XSL against the XML file to validate, resulting in a error report
			InputStream xslInputStream = new ByteArrayInputStream(
					((ByteArrayOutputStream)schematronXslResult.getOutputStream()).toByteArray());
			errorReportResult = new StreamResult(new ByteArrayOutputStream());
			Transformer schematronTransformer = 
				TransformerFactory.newInstance().newTransformer(new StreamSource(xslInputStream));
			schematronTransformer.transform(new StreamSource(xmlStream), errorReportResult);

		} catch (FileNotFoundException e) {
			String errorMsg = Messages.getString("SchematronValidationRule.4"); //$NON-NLS-1$
			XMLValidationError error = new XMLValidationError(errorMsg, this, e);
			validationErrorList.add(error);
			
			return validationErrorList;
			
		} catch (TransformerConfigurationException e) {
			String errorMsg = Messages.getString("SchematronValidationRule.5"); //$NON-NLS-1$
			XMLValidationError error = new XMLValidationError(errorMsg, this, e);
			validationErrorList.add(error);
			
			return validationErrorList;
			
		} catch (TransformerException e) {
			String errorMsg = Messages.getString("SchematronValidationRule.6"); //$NON-NLS-1$
			XMLValidationError error = new XMLValidationError(errorMsg, this, e);
			validationErrorList.add(error);
			
			return validationErrorList;			
		}
			
		// Parse the error report
		BufferedReader errorReport = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(
				((ByteArrayOutputStream)errorReportResult.getOutputStream()).toByteArray())));
		
	   try {
		   // Each line of the report contains one error - report is empty if there's no error
			String errorReportLine = errorReport.readLine();
			while (errorReportLine != null) {
				// Found an error, add it to the list
				XMLValidationError validationError = new XMLValidationError(errorReportLine, -1, -1, this,
						XMLValidationError.ERROR);
				validationErrorList.add(validationError);
				
				errorReportLine = errorReport.readLine();
			}
		   
	   } catch (IOException e) {
			String errorMsg = Messages.getString("SchematronValidationRule.7"); //$NON-NLS-1$
			XMLValidationError error = new XMLValidationError(errorMsg, this, e);
			validationErrorList.add(error);
			
			return validationErrorList;   
	   }
	   
	   // Return the list of errors (an empty array will be returned if no errors)
	   return validationErrorList;
	}
	
	public File getSchematronRulesFile() {
		return schematronRulesFile;
	}
}