/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.examples.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import org.eclipse.cosmos.me.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.sdd.common.validation.XML_DAS;
import org.eclipse.cosmos.me.sdd.common.validation.exception.XMLValidationException;
import org.eclipse.cosmos.me.sdd.common.validation.plugin.SchemaValidationRule;
import org.eclipse.cosmos.me.sdd.common.validation.plugin.XML_DASImpl;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

/*
 *  XML_DAS Sample
 *  
 *  Eric S Rose, esrose@us.ibm.com
 *  
 *  The XML Data Access Service (DAS) provides an interface for loading XML into a Document Object Model (DOM)
 *  Document, validating the contents of the Document against a set of ValidtionRules and writing a Document
 *  back out in XML.  The XML_DAS implementation provided uses InputStreams and OutputStream as the back end
 *  XML store.  It is intended for use with any XML for which there is an associated Schema.  The SDD_DAS
 *  extends the XML_DAS and provides functionality specifically for SDD's, which makes it more convenient to use
 *  than the generic XML_DAS for that purpose.
 *  
 *  This sample shows how to perform the following steps using the provided XML_DAS implementation:
 *  
 *  - Load the contents of an SDD XML file into a Document
 *  - Make a change to the Document
 *  - Validate the Document against an XML Schema
 *  - Write the Document back out to a (different) XML file
 */


public class XML_DAS_Sample {
    // Use platform independent file separator
    private static final String FSEP = System.getProperty("file.separator");

    // Path to the location of files needed for the sample to run
    private static final String filesPath = "src" + FSEP + "org" + FSEP + "eclipse" + FSEP + "cosmos" + FSEP + "me"
    	+ FSEP + "internal" + FSEP + "deployment" + FSEP + "sdd" + FSEP + "examples" + FSEP + "common" + FSEP
    	+ "validation" + FSEP;
	
	// This sample depends on the following files
	private static final String XML_INPUT_FILE = filesPath + "cd_catalog.xml";
	private static final String CD_SCHEMA_FILE = filesPath + "CompactDiscCatalog.xsd";
	
	// This file is created by the sample
	private static final String XML_OUTPUT_FILE = filesPath + "cd_catalog-rounded.xml";
	
	// Used when constructing a SchemaValidationRule
	private static final Collection<File> schemaFiles = Arrays.asList(new File[] {new File(CD_SCHEMA_FILE)});
	
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Delete the output file for a fresh start
		prepFiles();
		
		// Create an XML_DAS instance to use for reading and writing the XML file
		XML_DAS cdCatalogDAS = new XML_DASImpl();
		
		// Load the XML into a Document
		Document cdCatalogDoc = null;
		try {
			cdCatalogDoc = cdCatalogDAS.loadDocument(new FileInputStream(XML_INPUT_FILE));
		} catch (FileNotFoundException e) {
			// Thrown if the XML file is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (IOException e) {
			// Thrown if the XML file is present but unreadable
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (XMLValidationException e) {
			// Thrown if the XML is not well-formed
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		}
		
		/*
		 * The sample XML file contains sequences of elements that describe entries in a CD catalog. For example:
		 * 
		 *	<cd>
		 *		<title>The Very Best of</title>
		 * 		<artist>Cat Stevens</artist>
		 *		<country>UK</country>
		 *		<company>Island</company>
		 *		<price>8.90</price>
		 *		<year>1990</year>
		 *	</cd>
		 *
		 * We'll round each price up to the next highest whole dollar and write out the new XML to a different file.
		 */
		// Round up the CD prices
		roundUpPrices(cdCatalogDoc);
		
		/*
		 * Now that we've made a change, let's make sure the Document is still valid.  We'll do that by
		 * creating a SchemaValidationRule that validates against the CD Catalog Schema.  We'll add the rule
		 * to our DAS and let the DAS validate the Document.
		 */
		// Create the ValidationRule...
		SchemaValidationRule schemaRule = null;
		try {
			schemaRule = new SchemaValidationRule(schemaFiles);
		} catch (FileNotFoundException e) {
			// Thrown if any of the schema files are missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (IOException e) {
			// Thrown if any of the schema files are unreadable
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		}
		// ...and add it to the DAS
		cdCatalogDAS.addValidation(schemaRule);
		
		// Validate the updated Document
		Collection<XMLValidationError> validationErrors = cdCatalogDAS.validate(cdCatalogDoc);
		
		if (validationErrors.size() > 0) {
			// Document isn't valid
			for (XMLValidationError error : validationErrors) {
				System.err.println(error.getErrorMessage());
			}
			// No point in continuing - bail out
			System.exit(1);
		}
		
		// If we get here, the Document is valid.
		// Write it to a different XML file by passing the XML_DAS an OutputStream pointing to a different file
		try {
			cdCatalogDAS.saveDocument(cdCatalogDoc, new FileOutputStream(XML_OUTPUT_FILE), null, false);
			/*
			 * NOTE:
			 * We could have saved the DataGraph to the file and performed validation in a single step by
			 * calling XML_DAS.saveDataGraph() without the last parameter, which in turn calls the method
			 * above with a last parameter of true.
			 */
		} catch (XMLValidationException e) {
			// Since we didn't validate during save, this exception should never be thrown
		} catch (IOException e) {
			// Thrown if the output file isn't writable
			e.printStackTrace();
		}
		
		/*
		 * Check the output file to see the results of the Document changes
		 */
	}
	
	private static void prepFiles() {
		// Remove the output file if it exists
		File outputFile = new File(XML_OUTPUT_FILE);
		outputFile.delete();
	}
	
	private static void roundUpPrices(Document catalog) {
		// Loop through all cd elements
		NodeList nodes = catalog.getDocumentElement().getChildNodes();
		for (int i = 0; i < nodes.getLength(); i++) {
			// Find the price element for this CD
			NodeList children = nodes.item(i).getChildNodes();
			for (int j = 0; j < children.getLength(); j++) {
				if ("price".equals(children.item(j).getNodeName())) {
					// Round up the price
					double oldPrice = Double.parseDouble(children.item(j).getTextContent());
					double newPrice = Math.ceil(oldPrice);
					
					children.item(j).setTextContent(new Double(newPrice).toString());
				}
			}
		}	
	}	
}
