/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

/**
 * An abstract representation of a declaration collection. See
 * {@link TypeDeclarationCollection} or {@link GroupDeclarationCollection} for
 * concrete implementations.
 * 
 * @author Ali Mehregani
 * @author David Whiteman
 */
public abstract class AbstractDeclarationCollection<T> {
	/**
	 * Serial version UID
	 */
	private static final long serialVersionUID = 5392889191351236628L;

	/**
	 * Indexes declarations by namespace and local name
	 */
	private Map<QName, T> uriNameIndex;

	/**
	 * Internal implementation of the collection, encapsulated to allow for flexibility.
	 */
	private List<T> collection;

	public AbstractDeclarationCollection() {
		uriNameIndex = new Hashtable<QName, T>();
		collection = new ArrayList<T>();
	}

	/**
	 * Answer an iterator for the receiver.
	 * 
	 * @return an iterator over the collection
	 */
	public Iterator<T> iterator() {
		// Only need to iterate over the internal collection
		return collection.iterator();
	}

	/**
	 * Add the object to the collection, making sure the URI name index is
	 * also updated.
	 * 
	 * @see java.util.ArrayList#add(java.lang.Object)
	 * @param o the declaration to add
	 * @return whether the add was successful
	 */
	public boolean add(T o) {
		QName qName = ((AbstractDeclaration) o).getQName();
		if (qName != null) {
			uriNameIndex.put(qName, o);
		}
		return collection.add(o);
	}

	/**
	 * Returns a type declaration based on the uri and local name of the
	 * qualified name passed in.
	 * 
	 * @param name The qualified name
	 * @return An associated type declaration
	 */
	public T get(QName name) {
		return uriNameIndex.get(name);
	}
}
