/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.NamespaceContext;

import org.apache.xerces.xs.XSAnnotation;
import org.apache.xerces.xs.XSElementDeclaration;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure.IdentityConstraint;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * A utility class used for identity constraints
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintUtil
{
	/**
	 * Given an element declaration E, retrieve all identity constraints 
	 * that must be associated with E
	 * 
	 * @param elementDeclaration The element declaration E
	 * @param namespaceContext
	 */
	public static List<IdentityConstraint> retrieveConstraints(XSElementDeclaration elementDeclaration, NamespaceContext namespaceContext)
	{
		List<IdentityConstraint> identityConstraints = new ArrayList<IdentityConstraint>();
		if (elementDeclaration == null)
		{
			return identityConstraints;
		}
		
		Node[] nodes = SMLValidatorUtil.retrieveAnnotation(elementDeclaration, ISMLConstants.SML_URI, IdentityConstraintDataBuilder.IDENTITY_CONSTRAINTS);		
		for (int i = 0; i < nodes.length; i++)
		{			
			IdentityConstraint constraint = new IdentityConstraint();
			
			// Determine if this constraint is a reference
			constraint.setRef(retrieveAttribute(nodes[i], ISMLConstants.SML_URI, ISMLConstants.REF_ATTRIBUTE));						
			
			// Determine the name of the constraint
			constraint.setName(retrieveAttribute (nodes[i], ISMLConstants.SML_URI, ISMLConstants.NAME_ATTRIBUTE));
			
			// Determine the fields and the selectors
			Node[] fields = SMLValidatorUtil.findNodes(nodes[i], ISMLConstants.SML_URI, ISMLConstants.FIELD_ELEMENT);			
			Node[] selector = SMLValidatorUtil.findNodes(nodes[i], ISMLConstants.SML_URI, ISMLConstants.SELECTOR_ELEMENT);
			
			// Determine the type of the constraint
			String localName = nodes[i].getLocalName();
			byte type = ISMLConstants.KEY_ELEMENT.equals(localName) ? IdentityConstraint.KEY_TYPE :
						ISMLConstants.KEY_REF_ELEMENT.equals(localName) ? IdentityConstraint.KEY_REF_TYPE :
						IdentityConstraint.UNIQUE_TYPE;
						
			constraint.setNamespaceContext(namespaceContext);			
			
			// Set the type of the constraint
			constraint.setType(type);
			
			// Set the fields of the constraint			
			for (int j = 0; j < fields.length; j++)
			{		
				constraint.addField(retrieveAttribute(fields[j], ISMLConstants.SML_URI, ISMLConstants.XPATH_ATTRIBUTE));
			}
			
			// Set the selector			
			if (selector != null && selector.length > 0)
			{
				constraint.setSelector(retrieveAttribute(selector[0], ISMLConstants.SML_URI, ISMLConstants.XPATH_ATTRIBUTE));
			}
			
			// Set the refer
			constraint.setRefer(retrieveAttribute(nodes[i], ISMLConstants.SML_URI, ISMLConstants.REFER_ATTRIBUTE));
			constraint.setNamespace(elementDeclaration.getNamespace());
			
			identityConstraints.add(constraint);			
		}
		
		// Include the identity constraints of the substitution group element
		identityConstraints.addAll(retrieveConstraints (elementDeclaration.getSubstitutionGroupAffiliation(), namespaceContext));
		return identityConstraints;
	}
	
	
	/**
	 * Given node N, a namespace NS, and a local name L, retrieve the
	 * attribute from N with the namespace NS and local name L
	 * 
	 * @param node The node to retrieve the attribute from
	 * @param uri The namespace NS
	 * @param localName The local name L
	 * @return The attribute with namespace NS and local name L of N
	 */
	public static String retrieveAttribute(Node node, String uri, String localName)
	{
		NamedNodeMap attributes = node.getAttributes();
		if (attributes == null)
		{
			return null;
		}
		
		Node attribute = attributes.getNamedItem(localName);
		attribute = attribute == null ? attributes.getNamedItemNS(uri, localName) : attribute;		
		return attribute == null ? null : attribute.getNodeValue();
	}
	
	
	/**
	 * Given an element declaration E and a prefix p, using the annotation of
	 * E find the namespace p is bound to.
	 * 
	 * @param elementDeclaration The element declaration E
	 * @param prefix The prefix p
	 * @return The namespace of p based on the annotation of E
	 */
	public static String lookupNamespace(XSElementDeclaration elementDeclaration, String prefix)
	{
		Document document = SMLValidatorUtil.createDocument();
		elementDeclaration.getAnnotation().writeAnnotation(document, XSAnnotation.W3C_DOM_DOCUMENT);
		NamedNodeMap attributes = document.getFirstChild().getAttributes();
		Node namespace = attributes.getNamedItem(ISMLConstants.XML_NS_ATTRIBUTE + ":" + prefix);
		String result = null;
		return namespace == null || (result = namespace.getNodeValue()) == null ? 
				IValidationConstants.EMPTY_STRING : 
				result;
	}
}
