/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.application.impl;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


/**
 * This handler is used to determine type information of elements
 * 
 * @author Ali Mehregani
 */
public class TypeInfoHandler extends DefaultHandler
{
	/**
	 * The structure that is used to store the type information
	 */
	private Map<String, Map<String, String>> typeInfo;
		
	/**
	 * Stores the import statement of a schema file
	 * KEY = name space URI
	 * VALUE = The schema location
	 */
	private Map<String, String> importStatements;
	
	/**
	 * The prefix-name space map
	 * KEY = prefix
	 * VALUE = The name space URI
	 */
	private Map<String, String> prefixURIMap;

	/**
	 * The default name space URI
	 */
	private String defaultNamespace;

	/**
	 * Indicates the complex type name
	 */
	private String complexElementTypeName;

	/**
	 * The type hierarchy information
	 */
	private Map<String, Map<String, String>> hierarchyInfo;

	/**
	 * Stores the name attribute of the element declaration
	 */
	private String elementName;
	
	/**
	 * Stores the type attribute of the element declaration
	 */
	private String elementType;
	
	/**
	 * The constructor
	 * 
	 * @param typeInfo The type information
	 * @param hierarchyInfo The type hierarchy information
	 */
	public TypeInfoHandler(Map<String, Map<String, String>> typeInfo, Map<String, Map<String, String>> hierarchyInfo)
	{
		this.typeInfo = typeInfo;
		this.hierarchyInfo = hierarchyInfo;
		this.importStatements = new Hashtable<String, String>();
		this.prefixURIMap = new Hashtable<String, String>(); 
	}
	
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	@SuppressWarnings("unchecked")
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{		
		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			// The schema root element
			if (ISMLConstants.SCHEMA_ELEMENT.equals(localName))
			{
				// For every attribute
				for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
				{
					String attLocalName = attributes.getQName(i);
					if (!attLocalName.startsWith(ISMLConstants.XML_NS_ATTRIBUTE))
						continue;
					
					int colonInx = -1;
					String namespaceURI = attributes.getValue(attLocalName);
					String prefix = (colonInx = attLocalName.indexOf(':')) < 0 ? null : attLocalName.substring(colonInx + 1);
					
					if (namespaceURI != null)						
					{
						if (prefix != null)
							prefixURIMap.put(prefix, namespaceURI);
						else
							defaultNamespace = namespaceURI;
					}					
				}
				
				String targetNamespace = attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE);
				if (targetNamespace != null)
					defaultNamespace = targetNamespace;
			}			
			// Import statement
			else if (ISMLConstants.IMPORT_ELEMENT.equals(localName))
			{
				String namespace = attributes.getValue(ISMLConstants.NAMESPACE_ATTRIBUTE);
				String location = attributes.getValue(ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
				
				if (namespace != null && location != null)
					importStatements.put(namespace, location);
			}
			
			// Element declaration
			else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{			
				elementName = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);
				elementType = attributes.getValue(ISMLConstants.SCHEMA_URI, ISMLConstants.TYPE_ATTRIBUTE);
				elementType = elementType == null ? attributes.getValue(ISMLConstants.TYPE_ATTRIBUTE) : elementType;					
				
				String[] tokenizedType = tokenizeType(elementType);
				
				String namespace = tokenizedType[0];
				String type = tokenizedType[1];
				
				if (namespace != null)
				{
					Map<String, String> elementTypes = (Map<String, String>)SMLValidatorUtil.retrieveNestedMap(typeInfo, defaultNamespace, true);
					if (elementName != null)
						elementTypes.put(elementName, namespace + ":" + type);
				}					
			}
			
			// Complex type declaration
			else if (ISMLConstants.COMPLEXTYPE_ELEMENT.equals(localName))
			{
				complexElementTypeName = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);
			}
			
			// Complex content
			else if (ISMLConstants.EXTENSION_ELEMENT.equals(localName))
			{
				// This is the case where the complex content appears as a child of the element declaration 
				if (elementName != null && elementType == null)
				{
					// Fake an element type
					Map<String, String> elementTypes = (Map<String, String>)SMLValidatorUtil.retrieveNestedMap(typeInfo, defaultNamespace, true);
					complexElementTypeName = elementName+"Type";
					elementTypes.put(elementName, defaultNamespace + ":" + complexElementTypeName);				
				}
				
				String[] tonizedType = tokenizeType(attributes.getValue(ISMLConstants.BASE_ATTRIBUTE));				
				if (tonizedType[0] != null && tonizedType[1] != null)
				{
					Map<String, String> typeHirarchy = (Map<String, String>)SMLValidatorUtil.retrieveNestedMap(hierarchyInfo, defaultNamespace, true);
					typeHirarchy.put(complexElementTypeName, tonizedType[0] + ":" + tonizedType[1]);
				}
			}
		}
	}

	public String[] tokenizeType(String type)
	{
		if (type == null)
			return new String[2];
		
		int colonInx = -1;
		String prefix = (colonInx = type.indexOf(':')) < 0 ? null : type.substring(0, colonInx);
		String namespace = prefix == null ? defaultNamespace : (String)prefixURIMap.get(prefix);
		type = colonInx < 0 ? type : type.substring(colonInx + 1);   
		
		return new String[]{namespace, type};
	}
	
	public void endElement(String uri, String localName, String name) throws SAXException
	{
		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (ISMLConstants.COMPLEXTYPE_ELEMENT.equals(localName))
			{
				complexElementTypeName = null;
			}
			else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{
				elementName = null;
				elementType = null;
			}
		}
	}
	
	/**
	 * Returns the import statements discovered
	 * 
	 * @return The import statements
	 */
	public Map<String, String> getImportStatements()
	{
		return importStatements;
	}
}
