/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.editor;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * An abstract form page that all other form pages are expected to
 * extend.
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractFormPage extends FormPage implements ModifyListener
{	
	static String CONTEXT_ID_SMLIF_EDITOR = "org.eclipse.cosmos.rm.smlif.smlEditor";
	/**
	 * Indicates whether this form should be enabled
	 */
	private boolean enabled;

	public AbstractFormPage(FormEditor editor, String id, String title)
	{
		super(editor, id, title);
		enabled = true;
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormPage#createFormContent(org.eclipse.ui.forms.IManagedForm)
	 */
	protected void createFormContent(IManagedForm managedForm)
	{
		ScrolledForm form = managedForm.getForm();		
		form.setText(getTitle());
		
		Composite parent = form.getBody();
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;		
		parent.setLayout(gridLayout);
		parent.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		SashForm sashForm = ((SMLIFEditor)getEditor()).getExtendedToolkit().createSashForm(parent, SWT.HORIZONTAL);
		sashForm.setLayout(new GridLayout());
		sashForm.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				
		Composite leftColumn = createColumn(sashForm);
		Composite rightColumn = createColumn(sashForm);
		setWeight(sashForm);
		
		createLeftColumn(leftColumn);
		createRightColumn(rightColumn);				
		
		form.setEnabled(isEnabled());
	}	
	
	protected void setWeight(SashForm sashForm)
	{
		sashForm.setWeights(new int[]{5, 5});
	}

	public void setErrorMessage(String message) {
		ScrolledForm form = getManagedForm().getForm();
		if (message == null) {
			form.setMessage(null, IMessageProvider.NONE);
		} else {
			form.setMessage(message, IMessageProvider.ERROR);
		}
	}
	
	/**
	 * Create the left column content of the page
	 * 
	 * @param parent The parent composite
	 */
	protected abstract void createLeftColumn(Composite parent);
	
	
	/**
	 * Create the right column content of the page
	 * 
	 * @param parent The parent composite
	 */
	protected abstract void createRightColumn(Composite parent);


	/**
	 * Creates a composite that corresponds to the a column in the given parent.
	 * @param toolkit 
	 * 
	 * @param parent The parent composite
	 * @return Composite The column composite
	 */
	protected Composite createColumn(Composite parent)
	{
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, CONTEXT_ID_SMLIF_EDITOR);
		Composite column = getManagedForm().getToolkit().createComposite(parent);
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.verticalSpacing = 10;
		gridLayout.marginHeight = 10;
		gridLayout.marginWidth = 0;
		column.setLayout(gridLayout);
		column.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		return column;
	}
	
	
	/**
	 * Create and return a section
	 * 
	 * @return A section
	 */
	protected Section createSection(Composite parent)
	{
		Section section = getManagedForm().getToolkit().createSection(parent, Section.TITLE_BAR );
		GridData td = new GridData(SWT.FILL, SWT.FILL, true, true);
		section.setLayoutData(td);
		return section;
	}
	
	
	/**
	 * Create a label and a text field.  The controls will be
	 * layedout horizonally:   <label>  <text>
	 * 
	 * @param parent The parent composite
	 * @param label The label text
	 * @param multiLine Whether the text box should be multi-line
	 * @param initialValue The initial value of the text box
	 * @param modifyListener The modify listener that will be registered with the text box
	 */
	protected Text createTextField(Composite parent, String label, boolean multiLine, String initialValue, ModifyListener modifyListener)
	{
		FormToolkit toolkit = getManagedForm().getToolkit();
		Label lbl = toolkit.createLabel(parent, label);
		GridData lblGD = new GridData(SWT.DEFAULT, SWT.TOP, false, false);
		lbl.setLayoutData(lblGD);
		
		Text text = toolkit.createText(parent, initialValue, multiLine ? SWT.BORDER | SWT.V_SCROLL | SWT.WRAP : SWT.NONE);
		GridData gd = new GridData(SWT.FILL, SWT.DEFAULT, true, false);
				
		if (multiLine)
		{
			gd.widthHint = 200;
			gd.heightHint = 200;
		}
			
		
		text.setLayoutData(gd);
		if (initialValue != null)
			text.setText(initialValue);
		
		if (modifyListener != null)
			text.addModifyListener(modifyListener);		
		
		return text;
	}


	/**
	 * Equivalent to createTextField(parent, label, false, initialValue, modifyListener)
	 */
	protected Text createTextField(Composite parent, String label, String initialValue, ModifyListener modifyListener)
	{
		return createTextField (parent, label, false, initialValue, modifyListener);
	}
	
	
	/**
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{
		if (!isDirty())
			getManagedForm().dirtyStateChanged();
	}
	
	/**
	 * Set the appropriate properties of the section
	 * 
	 * @param section The section
	 * @param client The section client
	 * @param title The title of the section
	 */
	protected void makeSectionReady(Section section, Composite client, String title)
	{
		section.setClient(client); 
		section.setText(title);
		getManagedForm().getToolkit().createCompositeSeparator(section);
		section.setExpanded(true);
	}
	
	/**
	 * Used to set the enable status of this page
	 * 
	 * @param enabled The enable status
	 */
	protected void setEnabled(boolean enabled)
	{
		if (getManagedForm() != null)
		{
			getManagedForm().getForm().setEnabled(false);
		}
		this.enabled = enabled; 
	}
	
	
	/**
	 * Returns true if this form is suppose to be enabled; 
	 * false otherwise
	 * 
	 * @return true if this form is suppose to be enabled; false otherwise
	 */
	protected boolean isEnabled()
	{
		return enabled;
	}
	
	/**
	 * @see org.eclipse.ui.forms.editor.FormPage#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor)
	{
		/* Needs to be implemented by subclasses */
	}
	
	
	/**
	 * This method is invoked when the content of the SML-IF document
	 * has changed outside of the SML-IF document.
	 */
	protected abstract void updateContent();
	
}
