/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IQuery;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IQueryInputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Conversions between XML query structures and Java object graphs
 * 
 * 
 * @author David Whiteman
 */
public class QueryInputTransformer {

	/**
	 * Convert an IQuery into the XML syntax for queries described in the CMBDf spec
	 * 
	 * @param query
	 * @return an input stream on the XML containing the <query> structure
	 */
	public static InputStream transform(IQuery query) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) query);
	}

	/**
	 * Convert input stream containing an XML structure for a CMDBf query into a
	 * Java object graph
	 * 
	 * @param xmlInputStream
	 * @return IQuery object structure representing a query
	 * @throws TransformerException
	 */
	public static IQuery transform(InputStream xmlInputStream) throws TransformerException {
		return transform (xmlInputStream, null, null);	
	}

	public static IQuery transform(InputStream xmlInputStream, IQueryInputArtifactFactory artifactFactory) throws TransformerException {
		return transform (xmlInputStream, artifactFactory, null);	
	}

	
	/**
	 * Convert input stream containing an XML structure for a CMDBf query into a
	 * Java object graph
	 * 
	 * @param reader
	 * @return IQuery object structure representing a query
	 * @throws TransformerException
	 */
	public static IQuery transform(Reader reader) throws TransformerException {
		return transform (reader, null, null);	
	}

	public static IQuery transform(Reader reader, IQueryInputArtifactFactory artifactFactory) throws TransformerException {
		return transform (reader, artifactFactory, null);	
	}

	/**
	 * Convert input stream containing an XML structure for a CMDBf query into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformOptions a Map of options that affect default behavior of the transformer
	 * @return IQuery object structure representing a query
	 * @throws TransformerException
	 */
	public static IQuery transform(InputStream xmlInputStream, IQueryInputArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException
	{
		return transformOptions == null ? 
				(IQuery) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new QueryInputStreamHandler(artifactFactory)):
				(IQuery) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new QueryInputStreamHandler(artifactFactory), transformOptions);	
	}
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf query into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformOptions a Map of options that affect default behavior of the transformer
	 * @return IQuery object structure representing a query
	 * @throws TransformerException
	 */
	public static IQuery transform(Reader reader, IQueryInputArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException 
	{
		return transformOptions == null ? 
				(IQuery) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new QueryInputStreamHandler(artifactFactory)) :
				(IQuery) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new QueryInputStreamHandler(artifactFactory), transformOptions);	
	}

}
