/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.dialogs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.cosmos.rm.internal.smlif.SMLImages;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;

/**
 * A dialog used to list relevant resources based on the set
 * content type.
 * 
 * @author Ali Mehregani
 */
public class ResourceSelectionDialog extends Dialog
{
	/**
	 * The shell of the dialog
	 */
	private Shell sShell;
	
	/**
	 * The selected files
	 */
	private IFile[] selectedFiles;

	/**
	 * The content type
	 */
	private int contentType;
	
	/**
	 * Indicates whether multi-selection should
	 * be allowed
	 */
	private boolean multiSelection;
	
	/**
	 * The title of the dialog
	 */
	private String title;
	
	/**
	 * The message of the dialog
	 */
	private String message;
	
	
	/**
	 * The constructor
	 * 
	 * @param parent The parent shell
	 * @param contentType The content type that this resource selection dialog should
	 * be based on.  The contentType should be one of the following constatns:
	 * <ul>
	 * 	<li>{@link ISMLConstants#TYPE_DEFINITION}</li>
	 * 	<li>{@link ISMLConstants#TYPE_INSTANCE}</li>
	 * 	<li>{@link ISMLConstants#TYPE_SMLIF}</li>
	 * </ul> 
	 */	
	public ResourceSelectionDialog(Shell parent, int contentType)
	{
		super(parent);
		sShell = new Shell(getParent(), SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		this.contentType = contentType;
		this.multiSelection = false;
	}

	public IFile[] open()
	{		
		final Point shellBounds = new Point (370, 456);		
		sShell.setText(getTitle() == null ? SMLMessages.dialogResourceTitle : getTitle());
		sShell.setLayout(new GridLayout());
		sShell.setSize(shellBounds);
		sShell.setLocation(getParent().getDisplay().getBounds().width / 2 - (shellBounds.x / 2), getParent().getClientArea().height / 2 - (shellBounds.y / 2));
		sShell.setImage(SMLImages.INSTANCE.getImage(SMLImages.RESOURCE));

		createContent();
		sShell.open();    		
		Display display = getParent().getDisplay();
		
		while (!sShell.isDisposed()) 
		{
			if (!display.readAndDispatch()) display.sleep();
		}
		
		dispose();
		return selectedFiles;
	}

	public void dispose()
	{
		if (sShell != null)
		{
			sShell.dispose();
			sShell = null;
		}
	}
	
	private void createContent()
	{
		Composite composite = new Composite(sShell, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		Label selectLabel = new Label(composite, SWT.NONE);
		selectLabel.setText(getMessage() == null ? (isMultiSelection() ? SMLMessages.dialogResourceQuestion : SMLMessages.dialogResourceQuestion1) : getMessage());
		selectLabel.setLayoutData(new GridData(SWT.FILL, SWT.NONE, true, false));
		
		final Tree resourceTree = new Tree(composite, SWT.BORDER | (isMultiSelection() ? SWT.MULTI : SWT.NONE));
		resourceTree.setLayout(new GridLayout());
		resourceTree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		final TreeViewer resourceTreeViewer = new TreeViewer(resourceTree);
		resourceTreeViewer.setContentProvider(new ResourceTreeContentProvider(contentType));
		resourceTreeViewer.setLabelProvider(new ResourceTreeLabelProvider());
		resourceTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
		
		
		Composite buttonComposite = new Composite(sShell, SWT.NONE);
		buttonComposite.setLayout(new GridLayout(2, true));
		buttonComposite.setLayoutData(new GridData(SWT.RIGHT, SWT.DEFAULT, true, false));
		
		final Button okButton = createButton(buttonComposite, SMLMessages.commonOK);
		final Button cancelButton = createButton(buttonComposite, SMLMessages.commonCancel);
		
		final SelectionListener buttonListener = new SelectionAdapter()
		{			
			public void widgetSelected(SelectionEvent e)
			{
				if (e.widget == okButton)
				{
					StructuredSelection selection = (StructuredSelection)resourceTreeViewer.getSelection();
					Object[] selectedObjects = selection.toArray();
					List<IFile> fileList = new ArrayList<IFile>();
					for (int i = 0; i < selectedObjects.length; i++)
					{
						if (selectedObjects[i] instanceof IFile)
						{
							fileList.add((IFile)selectedObjects[i]);
						}						
					}
					
					selectedFiles = fileList.toArray(new IFile[fileList.size()]);
				}
				else
				{
					selectedFiles = new IFile[0];
				}
				sShell.close();
			}
		};
		
		resourceTreeViewer.addSelectionChangedListener(new ISelectionChangedListener()
		{
			public void selectionChanged(SelectionChangedEvent event)
			{				
				if (event.getSelection().isEmpty() || !(event.getSelection() instanceof StructuredSelection))
				{
					okButton.setEnabled(false);
				}
				
				for (Iterator<?> selectedItems = ((StructuredSelection)event.getSelection()).iterator(); selectedItems.hasNext();)
				{
					if (selectedItems.next() instanceof IFile)
					{
						okButton.setEnabled(true);
						return;
					}
				}
				
				okButton.setEnabled(false);
			}
		});
		
		resourceTree.addSelectionListener(new SelectionAdapter()
		{
			public void widgetDefaultSelected(SelectionEvent e)
			{
				Event event = new Event();
				event.widget = okButton;
				buttonListener.widgetSelected(new SelectionEvent(event));
			}
		});
		
		okButton.addSelectionListener(buttonListener);
		cancelButton.addSelectionListener(buttonListener);
	}
	
	
	private Button createButton(Composite parent, String text)
	{
		Button button = new Button (parent, SWT.NONE);
		button.setText(text);
		GridData buttonGridData = new GridData();
		buttonGridData.widthHint = 80;
		button.setLayoutData(buttonGridData);
		
		return button;
	}

	/**
	 * @return the multiSelection
	 */
	public boolean isMultiSelection()
	{
		return multiSelection;
	}

	/**
	 * @param multiSelection the multiSelection to set
	 */
	public void setMultiSelection(boolean multiSelection)
	{
		this.multiSelection = multiSelection;
	}

	/**
	 * @return the title
	 */
	public String getTitle()
	{
		return title;
	}

	/**
	 * @param title the title to set
	 */
	public void setTitle(String title)
	{
		this.title = title;
	}
	
	/**
	 * @return the message
	 */
	public String getMessage()
	{
		return message;
	}

	/**
	 * @param message the message to set
	 */
	public void setMessage(String message)
	{
		this.message = message;
	}
	
	private static class ResourceTreeContentProvider implements ITreeContentProvider
	{
		/**
		 * The input
		 */
		private IWorkspaceRoot input;

		/**
		 * The content type
		 */
		private int contentType;
		
		public ResourceTreeContentProvider(int contentType)
		{
			this.contentType = contentType;
		}
		
		public Object[] getChildren(Object parent)
		{
			try
			{
				List<IResource> children = new ArrayList<IResource>();
				IResource[] resources = ((IContainer)parent).members();
				for (int i = 0; i < resources.length; i++)
				{
					if (resources[i] instanceof IContainer)
					{
						children.add(resources[i]);
					}
					else if (resources[i] instanceof IFile && contentType == SMLValidatorUtil.identifyDocumentType((IFile)resources[i]))
					{
						children.add(resources[i]);
					}
				}
				
				return children.toArray();
			} 
			catch (CoreException e)
			{				
			} 
			catch (IOException e)
			{				
			}
			return new Object[0];
		}

		public Object getParent(Object element)
		{			
			return null;
		}

		public boolean hasChildren(Object element)
		{
			if (element instanceof IContainer)
				return true;
			
			return false;
		}

		public Object[] getElements(Object inputElement)
		{
			if (input == null)
				return new Object[0];
			
			return input.getProjects();			
		}

		public void dispose()
		{		
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
			this.input = (IWorkspaceRoot)newInput;
		}
		
	}
	
	
	private static class ResourceTreeLabelProvider implements ILabelProvider
	{

		public Image getImage(Object element)
		{
			if (element instanceof IProject)
				return SMLImages.INSTANCE.getImage(SMLImages.PROJECT);
			else if (element instanceof IContainer)
				return SMLImages.INSTANCE.getImage(SMLImages.FOLDER);			
			return SMLImages.INSTANCE.getImage(SMLImages.DOCUMENTS);
		}

		public String getText(Object element)
		{
			return ((IResource)element).getName();
		}

		public void addListener(ILabelProviderListener listener)
		{		
		}

		public void dispose()
		{
		}

		public boolean isLabelProperty(Object element, String property)
		{			
			return false;
		}

		public void removeListener(ILabelProviderListener listener)
		{
		}
		
	}

	
}
