/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.internal.cmdbf.services.transform.artifacts;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IAdditionalRecordType;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecord;

public abstract class GraphElement implements IGraphElement, IXMLWritable {

	private List<IRecord> records = new ArrayList<IRecord>();
	private List<IInstanceId> instanceIds = new ArrayList<IInstanceId>();
	private List<IAdditionalRecordType> additionalRecordTypes = new ArrayList<IAdditionalRecordType>();

	/**
	 * Answer the name for the XML element representation for the receiver
	 * @return
	 */
	protected abstract String getElementName();

	// To be overridden by subclasses that have additional subelements to output
	protected void outputAdditionalSubelements(StringWriter writer, int i) {
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(CMDBfServicesUtil.beginTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + getElementName())+ICMDBfServicesConstants.nl);
		
		outputAdditionalSubelements(writer, indent+1);
		
		CMDBfServicesUtil.outputListToXML(writer, indent+1, primGetRecords());
		CMDBfServicesUtil.outputListToXML(writer, indent+1, primGetInstanceIds());
		CMDBfServicesUtil.outputListToXML(writer, indent+1, primGetAdditionalRecordTypes());
		
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + getElementName())+ICMDBfServicesConstants.nl);
	}

	public IRecord[] getRecords() 
	{
		return (IRecord[]) primGetRecords().toArray(new IRecord[records.size()]);
	}
	
	public void setRecords(IRecord[] records)
	{
		this.records.clear();
		for (int i = 0; i < records.length; i++)
		{
			addRecord(records[i]);
		}
	}
	
	private List<IRecord> primGetRecords() {
		return records;
	}

	public IInstanceId[] getInstanceIds() {
		return instanceIds.toArray(new IInstanceId[instanceIds.size()]);
	}

	public IAdditionalRecordType[] getAdditionalRecordTypes() 
	{
		return additionalRecordTypes.toArray(new IAdditionalRecordType[additionalRecordTypes.size()]);
	}

	public void addRecord(IRecord record) {
		primGetRecords().add(record);
	}

	public void removeRecord(IRecord record)
	{
		primGetRecords().remove(record);
	}
	
	public void addAdditionalRecordType(IAdditionalRecordType additionalRecordType) {
		primGetAdditionalRecordTypes().add(additionalRecordType);
	}

	private List<IAdditionalRecordType> primGetAdditionalRecordTypes() {
		return additionalRecordTypes;
	}

	public void addInstanceId(IInstanceId instanceId) {
		primGetInstanceIds().add(instanceId);
	}

	private List<IInstanceId> primGetInstanceIds() {
		return instanceIds;
	}

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime *
				result +
				((additionalRecordTypes == null) ? 0 : additionalRecordTypes
						.hashCode());
		result = prime * result +
				((instanceIds == null) ? 0 : instanceIds.hashCode());
		result = prime * result + ((records == null) ? 0 : records.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final GraphElement other = (GraphElement) obj;
		if (additionalRecordTypes == null) {
			if (other.additionalRecordTypes != null)
				return false;
		} else if (!additionalRecordTypes.equals(other.additionalRecordTypes))
			return false;
		if (instanceIds == null) {
			if (other.instanceIds != null)
				return false;
		} else if (!instanceIds.equals(other.instanceIds))
			return false;
		if (records == null) {
			if (other.records != null)
				return false;
		} else if (!records.equals(other.records))
			return false;
		return true;
	}

}
