/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.core;

import java.util.HashMap;
import java.util.Map;

import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSTerm;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;


/**
 * This is an abstract implementation of ISMLValidation.
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractSMLValidator extends AbstractValidator implements ISMLValidator
{	
	/**
	 * The scope of this validation class
	 */
	private String[] scope;

	/**
	 * Keeps track of constraint based on the qualified name of
	 * the element declaration.  The key is always a namespace:localName
	 * and the value is determined by the subclass
	 */
	private Map<String, Object> constraintMap;
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.ISMLValidator#getScope()
	 */
	public String[] getScope()
	{
		return scope;
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.ISMLValidator#setScope(java.lang.String[])
	 */
	public void setScope(String[] scope)
	{
		this.scope = scope;
	}


	protected boolean shouldAbortOnError() 
	{
		ValidationEvent validationEvent = new ValidationEvent();
		fireValidationEventOccurred(validationEvent);
		return validationEvent.abortOnError();
	}
	
	
	/**
	 * Used by the target and identity constraint validator to ensure
	 * the consistency of constraints according to the following rule
	 * <p>
	 *  If two element declarations E1 and E2 have the same {namespace name} and 
	 *  {name} and they are both contained (directly, indirectly, or implicitly) 
	 *  in a content model of a complex type, then E1 and E2 MUST have the same set 
	 *  of {SML identity-constraints definitions} and {SML target-constraints definitions}.
     * </p>
	 * 
	 * @return true if the validation passes; false otherwise
	 */
	protected boolean checkForInvalidDerivations() 
	{		
		XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
		if (xsModel == null)
		{
			return true;
		}
		
		boolean status = true;
		XSNamedMap typeDefinitions = xsModel.getComponents(XSConstants.TYPE_DEFINITION);
		constraintMap = new HashMap<String, Object>();
		for (int i = 0, typeDefCount = typeDefinitions.getLength(); i < typeDefCount; i++)
		{
			XSTypeDefinition typeDefX = (XSTypeDefinition) typeDefinitions.item(i);
			
			// do we need to exclude typeDefX == typeDef?  Or can a type
			// contain a reference to another instance of itself?
			if (typeDefX.getTypeCategory() != XSTypeDefinition.COMPLEX_TYPE) 
			{
				continue;
			}
			
			XSComplexTypeDefinition complexTypeDefX = (XSComplexTypeDefinition) typeDefX;
			XSParticle particle = complexTypeDefX.getParticle();
			if (particle == null) 
			{
				continue;
			}
			
			status = status && isParticleValid(particle, complexTypeDefX);
			if (shouldAbortOnError() && !status) 
			{
				constraintMap = null;
				return false;				
			}
		}
		
		constraintMap = null;
		return status;
	}
	
	
	/**
	 * Check the particles (model groups, elements, etc.) of the complex type
	 * for invalid target type derivations, invalid target required derivations,
	 * or invalid target element derivations.  Also check for inconsistent declarations
	 * when there are two like named elements with different targets contained by the same
	 * complex type.
	 * 
	 * @param particle
	 * @param typeDefinition
	 * @return true if the particle is valid for these conditions
	 */
	protected boolean isParticleValid(XSParticle particle, XSComplexTypeDefinition typeDefinition) 
	{
		XSTerm term = particle.getTerm();
		if (term instanceof XSElementDeclaration) 
		{
			XSElementDeclaration elementDeclarationToCheck = (XSElementDeclaration) term;
			String qualifiedName = SMLValidatorUtil.createQualifiedName(elementDeclarationToCheck);
			Object constraint = constraintMap.get(qualifiedName);
			return constraint == null ? 
						handleNewElement(elementDeclarationToCheck, typeDefinition) :
						handleDiscoveredElement(elementDeclarationToCheck, typeDefinition, constraint);
			
		}
		
		// Elements can be embedded in model groups
		if (term instanceof XSModelGroup) 
		{
			boolean valid = true;
			XSObjectList particles = ((XSModelGroup)term).getParticles();
			for (int j = 0, particleCount = particles.getLength(); j < particleCount /*&& (!found2)*/; j++) 
			{
				XSParticle iParticle = (XSParticle) particles.item(j);
				valid = valid && isParticleValid(iParticle, typeDefinition);
				if (!valid && shouldAbortOnError()) 
				{
					return false;
				}
			}
			return valid;
		}
		return true;
	}

	
	/**
	 * Expected to be overwritten by subclasses.
	 * 
	 * @param elementDeclaration The element declaration 
	 * @param typeDefinition The type declaration
	 * @return false is returned to indicate a validation failure; otherwise
	 * true is returned
	 */
	protected boolean handleNewElement (XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition)
	{
		return true;
	}
	
	
	/**
	 * Expected to be overwritten by subclasses to handle an element
	 * declaration that's already been encountered
	 * 
	 * @param elementDeclaration The element declaration under analysis
	 * @param typeDefinition The type definition
	 * @param discoveredConstraint The discovered constraint
	 * @return false is returned to indicate a validation failure; otherwise
	 * true is returned
	 */
	protected boolean handleDiscoveredElement (XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition, Object discoveredConstraint)
	{
		return true;
	}


	/**
	 * @return the constraintMap
	 */
	protected Map<String, Object> getConstraintMap()
	{
		return constraintMap;
	}	
}
