/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeNode;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This abstract property extractor will extract a set of elements/attribute values
 * based on a set of conditions specified (e.g. the value of the name attribute for elements
 * that have acyclic set to true)
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractPropertyExtractor<T> extends AbstractDataBuilder<T>
{
	/**
	 * The phase that this builder should operate in
	 */
	private byte phase;
	
	/**
	 * The element name
	 */
	protected String elementName;
	
	/**
	 * An array of String[] of length 3, where 
	 * String[0] = URI of the attribute name
	 * String[1] = attribute name
	 * String[2] = expected attribute value.
	 */
	private String[][] attributes;
	
	/**
	 * The key/value pair to store
	 * Must be of length 2
	 */
	private String[] keyValuePair;
	
	/**
	 * The list of elements stored that meet the required condition
	 */
	private Map<TypeNode, String> entries;
	
	/**
	 * Mapped structure
	 * KEY = The URI of a namespace (or an empty string for type nodes that don't have a uri)
	 * VALUE = A Map containing the following:
	 * 	KEY = A string indicating the type
	 * 	VALUE = The expected value stored
	 */
	private Map<String, Map<String, String>> mappedTypeNodes; 
	
	/**
	 * Constructor 
	 * 
	 * @param phase The phase that this builder should operate in (see ISMLConstants.*PHASE)
	 * @param elementName The expected element name
	 * @param attributes The expected attribute name and values
	 * @param keyValuePair The key/value pair to store.   Must be of length 2
	 */
	public AbstractPropertyExtractor(byte phase, String elementName, String[][] attributes, String[] keyValuePair)
	{
		this.phase = phase;
		this.elementName = elementName;
		this.attributes = attributes;
		this.keyValuePair = keyValuePair;
		entries = new Hashtable<TypeNode, String>();
	}
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{		
		super.startElement(uri, localName, qName, attributes);
		if (elementName != null && (!elementName.equals(localName)))
			return;
		
		if (this.attributes != null)
		{
			for (int i = 0; i < this.attributes.length; i++)
			{			
				int inx = attributes.getIndex(this.attributes[i][0], this.attributes[i][1]);
				if (inx < 0)
				{
					if (this.attributes[i][0].equals(uri))
						inx = attributes.getIndex(IValidationConstants.EMPTY_STRING, this.attributes[i][1]);
					if (inx < 0)
						return;
				}
				
				String currentAttributeValue = attributes.getValue(inx);	
				String[] tokenizedValue = SMLValidatorUtil.tokenizeName(currentAttributeValue);				
				if (this.attributes[i][2] != null && !this.attributes[i][2].equals(tokenizedValue[1]))
					return;
			}
		}
		
		String[] valueToStore = SMLValidatorUtil.tokenizeName(attributes.getValue(keyValuePair[0]));		
		String valueToStoreURI =  getValueToStoreNS() == null ? (valueToStore[0] == null ? super.getDefaultNamespace() : (String)super.getPrefixMap().get(valueToStore[0])) : getValueToStoreNS();
		TypeNode nodeType = new TypeNode(valueToStore[1], valueToStoreURI);
		if (valueToStore[1] == null || (valueToStore[1] != null && entries.get(nodeType) == null))
		{
			String key = keyValuePair[1] == null ? IValidationConstants.TRUE_VALUE : attributes.getValue(keyValuePair[1]);
			if (key != null)
				entries.put(nodeType, key);			
		}
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public Iterator<TypeNode> getIteratorDataStructure()
	{
		return entries.keySet().iterator();
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#getPhase()
	 */
	public byte getPhase() 
	{
		return phase;
	}

	/**
	 * Returns the map version of the entries stored.  The map object contains
	 * the following:
	 * KEY = The URI of a namespace (or an empty string for type nodes that don't have a uri)
	 * VALUE = A Map containing the following:
	 * 	KEY = A string indicating the type
	 * 	VALUE = The expected value stored
	 * 
	 * @return The map version of the entries stored
	 */
	protected Map<String, Map<String, String>> getMapDataStructure()
	{
		if (mappedTypeNodes == null)
		{
			mappedTypeNodes = new Hashtable<String, Map<String, String>>();
			for (Iterator<TypeNode> iter = entries.keySet().iterator(); iter.hasNext();)
			{
				TypeNode typeNode = iter.next();
				String uri = typeNode.getUri();
				uri = uri == null ? IValidationConstants.EMPTY_STRING : uri;
				String type = typeNode.getType();
				
				if (uri == null || type == null)
					continue;
				Map<String, String> nodesPerNS = mappedTypeNodes.get(uri);
				if (nodesPerNS == null)
				{
					nodesPerNS = new Hashtable<String, String>();
					mappedTypeNodes.put(uri, nodesPerNS);
				}
				
				nodesPerNS.put(type, entries.get(typeNode));
			}
		}
		return mappedTypeNodes;
	}
	
	
	protected String getValueToStoreNS()
	{
		return null;
	}
}
