/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;

/**
 * Represents a property with a set of operations.  This 
 * interface is used to compare properties for CMDBf queries that 
 * include a property value constraint.
 * 
 * 
 * @author Ali Mehregani
 */
public interface IProperty
{
	/**
	 * Represents an unknown type
	 */
	public final static int UNKNOWN = 0x00;
	
	/**
	 * Represents an boolean type
	 */
	public final static int BOOLEAN = 0x01;
	
	/**
	 * Represents an integer type
	 */
	public final static int INTEGER = 0x02;
	
	/**
	 * Represents an double type
	 */
	public final static int DOUBLE = 0x03;
	
	/**
	 * Represents an string type
	 */
	public final static int STRING = 0x04;
	
	
	/**
	 * Defined in terms of the XPath 2.0 value comparison operators
	 * "eq".
	 * 
	 * @param o The value to be compared against
	 * @param caseSensitive Indicates whether the comparison should be
	 * case sensitive
	 * @return true iff this property is equal to 'o'; false otherwise
	 */
	public boolean equals(Object o, boolean caseSensitive) throws CMDBfServiceException;
	
	/**
	 * Defined in terms of the XPath 2.0 value comparison operators
	 * "lt".
	 * 
	 * @param o The value to be compared against
	 * @return true iff this property is less than 'o'; false otherwise
	 */	
	public boolean less (Object o) throws CMDBfServiceException;
	
	/**
	 * Defined in terms of the XPath 2.0 value comparison operators
	 * "le".
	 * 
	 * @param o The value to be compared against
	 * @return true iff this property is less than or equal to 'o'; false otherwise
	 */	
	public boolean lessOrEqual(Object o) throws CMDBfServiceException;
	
	/**
	 * Defined in terms of the XPath 2.0 value comparison operators
	 * "gt".
	 * 
	 * @param o The value to be compared against
	 * @return true iff this property is greater than 'o'; false otherwise
	 */	
	public boolean greater (Object o) throws CMDBfServiceException;
	
	/**
	 * Defined in terms of the XPath 2.0 value comparison operators
	 * "ge".
	 * 
	 * @param o The value to be compared against
	 * @return true iff this property is greater than or equal to 'o'; false otherwise
	 */	
	public boolean greaterOrEqual(Object o) throws CMDBfServiceException;
	
	/**
	 * Maps to the XPath 2.0 function fn:contains().  This is only valid
	 * for properties that are of type String.
	 * 
	 * @param str The value to be compared against
	 * @param caseSensitive Indicates whether the comparison should be
	 * case sensitive
	 * @return true iff this string property contains 'str' as a substring; false otherwise
	 */	
	public boolean contains(String str, boolean caseSensitive) throws CMDBfServiceException;
		
	/**
	 * This operator is similar in functionality to the SQL LIKE clause.  The operator
	 * works like the equal operator with the inclusion of two special characters:
	 * <ul>
	 *  <li> the underscore ("_") which acts as a wild card for any single character </li>
	 *  <li> the percent sign ("%") which acts as a wild card for zero or more characters </li>
	 * </ul>
	 * Backslash ("\" can be used to escape any of the wild characters.
	 * 
	 * @param str The value to be compared against
	 * @param caseSensitive Indicates whether the comparison should be
	 * case sensitive
	 * @return true iff this property is like str; false otherwise
	 */
	public boolean like(String str, boolean caseSensitive) throws CMDBfServiceException;
	
	/**
	 * Indicates if this property is "nilled".  This is equivalent to applying the
	 * XPath 2.0 function "fn:nilled"
	 * 
	 * @return true iff the value of this property is nilled
	 */
	public boolean isNull() throws CMDBfServiceException;	
	
	
	/**
	 * Returns the property id associated with this property
	 * 
	 * @return The property id
	 */
	public IPropertyId getPropertyId();

	
	/**
	 * Sets the id of this property
	 *  
	 * @param propertyId The property id
	 */
	public void setPropertyId(IPropertyId propertyId);
}
