/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.resource;

import java.io.File;
import java.io.FileInputStream;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;

import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocumentFragment;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The file system implementation of an SML document.
 * 
 * @author Ali Mehregani
 */
public abstract class SMLFileDocument implements ISMLDocument
{
	/**
	 * A reference to the repository
	 */
	private ISMLRepository repository;
	
	/**
	 * The meta-data of this document
	 */
	private ISMLMetadata metadata;

	/**
	 * The last modified date of the file representing this document.
	 * This is used to cache the DOM content of the document.
	 */
	private long lastModifiedDate;
	
	/**
	 * The DOM content of this SML document.
	 */
	private Document domDocument;
		
	
	public SMLFileDocument(ISMLRepository repository)
	{
		this.repository = repository;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument#getMetadata()
	 */
	public ISMLMetadata getMetadata()
	{
		return metadata;
	}

	
	public void setMetadata (ISMLMetadata metadata)
	{
		this.metadata = metadata;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument#retrieveDocumentFragment(org.w3c.dom.Node, java.lang.String)
	 */
	public ISMLDocumentFragment retrieveDocumentFragment(NamespaceContext namespaceContext, Node contextNode, String xpathExpression) throws RepositoryOperationException
	{		
		if (getDOMDocument() == null)
			return null;
		
		XPathExpression fieldXPath;
		try
		{
			synchronized (SMLValidatorUtil.xpath)
			{
				if (namespaceContext != null)
					SMLValidatorUtil.xpath.setNamespaceContext(namespaceContext);
				fieldXPath = SMLValidatorUtil.xpath.compile(xpathExpression);			
				NodeList nodeList = (NodeList)fieldXPath.evaluate(contextNode == null ? getDOMDocument().getFirstChild() : contextNode, XPathConstants.NODESET);
				SMLDocumentFragment fragment = new SMLDocumentFragment();
				fragment.setParentDocument(this);
				fragment.setNodeList(nodeList);
				
				return fragment;
			}
		}
		catch (XPathExpressionException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		}
	}
	
	
	public boolean equals(Object o)
	{	
		if (!(o instanceof ISMLDocument))
			return false;
		
		ISMLMetadata metadata = ((ISMLDocument)o).getMetadata();
		if (this.metadata != null && this.metadata.getId() != null && metadata != null)
		{
			return new Path(this.metadata.getId()).equals(new Path(metadata.getId()));			
		}
		
		return this == o;
	}
	
	
	/**
	 * @return the repository
	 */
	public ISMLRepository getRepository()
	{
		return repository;
	}

	
	/**
	 * @return the lastModifiedDate
	 */
	protected long getLastModifiedDate()
	{
		return lastModifiedDate;
	}

	
	/**
	 * @return the domDocument
	 * @throws RepositoryOperationException 
	 */
	public Document getDOMDocument() throws RepositoryOperationException
	{
		/* Retrieve the content of the SML document */
		if (metadata == null || metadata.getId() == null)
			return null;	
		 
		File document = new File (metadata.getId());
		if (domDocument == null || document.lastModified() != lastModifiedDate)
		{
			if (!document.exists())
				return null;
			
			try
			{
				domDocument = CMDBfInternalUtility.domParseDocument(new FileInputStream(document));
			} 
			catch (Exception e)
			{
				throw new RepositoryOperationException(e.getLocalizedMessage(), e);
			}			
		}
		return domDocument;
	}
	
	/**
	 * Retrieves and returns the root node's attribute value with the 
	 * URI and local name passed in. 
	 * 
	 * @param uri The URI of the attribute
	 * @param localName The local name
	 * @return The attribute value
	 * @throws RepositoryOperationException In case of an error
	 */
	public String retrieveRootAttribute (String uri, String localName) throws RepositoryOperationException
	{
		Document document = getDOMDocument();
		
		Node node = document == null ? null : document.getFirstChild();
		return SMLRepositoryUtil.retrieveRootAttribute(node, uri, localName);
	}
}
