/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;

/**
 * This abstract class provides common operations required by
 * multiple validators 
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractValidator implements IValidator
{
	/**
	 * The validation output
	 */
	private IValidationOutput<String, Object> validationOutput;
		
	/**
	 * The validation listeners
	 */
	private Collection<IValidationListener> validationListeners;
	
	/**
	 * The progress monitor
	 */
	private IProgressMonitor monitor;
	
	/**
	 * Validation attribute
	 */
	private Map<String, Object> attributes;
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#reset()
	 */
	public void reset()
	{
		// Empty content - Subclasses are expected to
		// overwrite
	}	
	
	
	/**
	 * Read in common attributes that are shared between multiple
	 * validators.
	 * 
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#initialize(java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	public void initialize(Map<String,Object> validationAttribute)
	{
		setAttributes(validationAttribute);
		validationOutput = (IValidationOutput)validationAttribute.get(IValidator.ATTRIBUTE_OUTPUT);
		monitor = (IProgressMonitor)validationAttribute.get(IValidator.ATTRIBUTE_PROGRESS_MONITOR);
	}

	public void setAttributes(Map<String, Object> attributes) {
		this.attributes = attributes;
	}


	/**
	 * @return the validationOutput
	 */
	public IValidationOutput<String, Object> getValidationOutput()
	{
		return validationOutput;
	}

	/**
	 * @param validationOutput the validationOutput to set
	 */
	public void setValidationOutput(IValidationOutput<String, Object> validationOutput)
	{
		this.validationOutput = validationOutput;
	}

	public final void addValidationListener(IValidationListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		if (validationListeners == null) {
			validationListeners = new ArrayList<IValidationListener>();
		}
		if (!validationListeners.contains(listener)) {
			validationListeners.add(listener);
		}
	}
	
	protected final void fireValidationEventOccurred(final ValidationEvent event) {
		if (event == null) {
			throw new NullPointerException();
		}
		if (validationListeners != null) 
		{
			final Iterator<IValidationListener> listenerItr = validationListeners.iterator();
			while (listenerItr.hasNext()) 
			{
				final IValidationListener listener = listenerItr.next();
				listener.validationErrorOccurred(event);
			}
		}
	}

	/**
	 * Sets the task name for this validation
	 * phase.
	 * 
	 * @param name The task name
	 */
	protected void setTaskName (String name)
	{
		if (monitor == null)
		{
			return;
		}
		monitor.setTaskName(name);
	}


	/**
	 * @return the attributes
	 */
	public Map<String, Object> getAttributes()
	{
		return attributes;
	}
}
