/**********************************************************************
 * Copyright (c) 2008,2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SMLDocuments;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.FoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This builder is used to cache the definition and
 * instance documents.
 *
 * @author Ali Mehregani
 * @author John Arwe
 */
public class DocumentCacheBuilder extends AbstractDataBuilder<SMLDocuments>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".DocumentCacheBuilder";

	/**
	 * Flags used to indicate the position of the parser
	 */
	private boolean documentElementHit,
					dataElementHit,
					base64DataElementHit,
					instanceElementHit,
					definitionElementHit,
					baseURIElementHit,
					aliasElementHit;

	/**
	 * The current definition/instance element being processed
	 */
	private ElementModel currentElementModel;

	/**
	 * Indicates the location of the last data element reached
	 */
	private ElementLocation lastDataLocation;

	/**
	 * The SML-IF file being parsed
	 */
	private String smlifFile;

	/**
	 * A flag used to indicate if the validator is being executed in
	 * Eclipse
	 */
	private boolean inEclipse;

	/**
	 * The current alias
	 */
	private String currentAlias;

	/**
	 * A list of aliases associated with the current document
	 * being parsed.
	 */
	private List<String> currentAliases;

	/**
	 * The data structure
	 */
	private SMLDocuments smlDocuments;

	/**
	 * Stores all aliases
	 */
	private List<String> allAliases;

	/**
	 * The current base URI value being parsed
	 */
	private String currentBaseURI;


	/**
	 * Constructor
	 *
	 * @param smlifFile The SML-IF file being validated
	 */
	public DocumentCacheBuilder()
	{
		this.smlDocuments = new SMLDocuments();
		this.currentAliases = new ArrayList<String>();
		this.currentAlias = "";
		this.currentBaseURI = "";
		this.allAliases = new ArrayList<String>();
		addEvent(FoundationBuilder.EVENT_CHARACTER);
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		super.initialize(init);
		this.inEclipse = IValidator.VALUE_ENV_ECLIPSE.equals(init.get(IValidator.ATTRIBUTE_ENV));
		this.smlifFile = IValidator.VALUE_SML_IF.equals(init.get(IValidator.ATTRIBUTE_INPUT_TYPE)) ?
				(String)init.get(IValidator.ATTRIBUTE_INSTANCE) : null;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		super.startElement(uri, localName, qName, attributes);
		// The document and data or base64Data element is hit
		if (dataElementHit || base64DataElementHit)
		{
			if (!ISMLConstants.SCHEMATRON_URI.equals(uri) && lastDataLocation != null)
			{
				if (ISMLConstants.SCHEMA_ELEMENT.equals(localName))
				{
					Node remoteDocument = (Node)getInit().get(IValidator.ATTRIBUTE_REMOTE_DOCUMENT);
					String decodedDocument = (String)getInit().get(IValidator.ATTRIBUTE_DECODED_DATA);
					ElementSchemaModel schemaModel;
					if (remoteDocument != null) {
						schemaModel = new ElementSchemaModel(remoteDocument, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
					} else if (decodedDocument != null) {
						schemaModel = new ElementSchemaModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
						schemaModel.setSource(decodedDocument);
					} else {
						schemaModel = new ElementSchemaModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
					}
					schemaModel.setTargetNamespace(attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE));
					for (int i = 0; i < attributes.getLength(); i++)
					{
						String qNameAttribute = attributes.getQName(i);
						String valueAttribute = attributes.getValue(i);
						if (qNameAttribute.startsWith(ISMLConstants.XML_NS_ATTRIBUTE))
						{
							String prefix = retrievePrefix (qNameAttribute);

							if (prefix != null && valueAttribute != null)
							{
								schemaModel.getNamespaceContext().addEntry(prefix, valueAttribute);
							}
						}
					}
					currentElementModel = schemaModel;
				}
				else
				{
					Node remoteDocument = (Node)getInit().get(IValidator.ATTRIBUTE_REMOTE_DOCUMENT);
					String decodedDocument = (String)getInit().get(IValidator.ATTRIBUTE_DECODED_DATA);
					if (remoteDocument != null) {
						currentElementModel = new ElementModel(remoteDocument, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
					} else if (decodedDocument != null) {
						currentElementModel = new ElementModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
						currentElementModel.setSource(decodedDocument);
					} else {
						currentElementModel = new ElementModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
					}
				}

				lastDataLocation = null;
			}

			boolean importElement = false;
			if (ISMLConstants.SCHEMA_URI.equals(uri))
			{
				if (ISMLConstants.IMPORT_ELEMENT.equals(localName))
				{
					importElement = true;
					String namespace = attributes.getValue(ISMLConstants.NAMESPACE_ATTRIBUTE);
					if (namespace != null && currentElementModel instanceof ElementSchemaModel)
					{
						((ElementSchemaModel)currentElementModel).addImportedSchema(namespace);
					}
				}
			}


			boolean needSchemaLocation = importElement && (currentElementModel != null);
			boolean smlNamespace = false;
			if (importElement)
			{
				for (int i = 0; i < attributes.getLength(); i++)
				{
					needSchemaLocation = ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE.equals(attributes.getLocalName(i)) ? false : needSchemaLocation;
					smlNamespace = 	ISMLConstants.NAMESPACE_ATTRIBUTE.equals(attributes.getLocalName(i)) &&
									ISMLConstants.SML_URI.equals(attributes.getValue(i)) ?
											true : smlNamespace;
				}
			}

			// Add in schema location so SML validation works properly
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=177811
			if (smlNamespace && needSchemaLocation)
			{
				StringBuffer buffer = new StringBuffer(" "+ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
				buffer.append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(ISMLConstants.SML_URI).append(ISMLConstants.DOUBLE_QUOTE);
				currentElementModel.setAdditionalSchemaLocation(buffer.toString());
			}
		}


		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			// This test is only needed because of the associated JUnit
			if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = true;
			}

			else if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = true;
			}

			else if ((instanceElementHit || definitionElementHit) && ISMLConstants.BASE_URI_ELEMENT.equals(localName))
			{
				baseURIElementHit = true;
			}

			else if ((instanceElementHit || definitionElementHit) && ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = true;
			}

			// The document element is hit
			else if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = true;
				currentBaseURI = "";
			}

			// The data element is hit
			else if (documentElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = true;
				lastDataLocation = getLocator() == null ?
							new ElementLocation(null, 0, 0) :
							new ElementLocation(null, getLocator().getLineNumber() + 1, getLocator().getColumnNumber() + 1);
			}

			// base64Data element is hit
			else if (documentElementHit && ISMLConstants.BASE64DATA_ELEMENT.equals(localName))
			{
				base64DataElementHit = true;
				lastDataLocation = getLocator() == null ?
						new ElementLocation(null, 0, 0) :
						new ElementLocation(null, getLocator().getLineNumber() + 1, getLocator().getColumnNumber() + 1);
			}
		}
	}


	/**
	 * Given a qName of the form xmlns:prefix, return
	 * prefix
	 *
	 * @param qName The qualified name
	 * @return The prefix
	 */
	private String retrievePrefix(String qName)
	{
		int colonInx = qName.indexOf(":");
		return colonInx >= 0 && colonInx + 1 < qName.length() ? qName.substring(colonInx + 1) : null;
	}


	@Override
	public void characters(char[] chars, int start, int length) throws SAXException
	{
		if (baseURIElementHit)
		{
			currentBaseURI += new String(chars, start, length).trim();
		}
		if (aliasElementHit)
		{
			currentAlias += new String(chars, start, length).trim();
		}
	}


	private String elementFilePath()
	{
		// If we are processing an SML-IF file, getFilePath() will be null
		return smlifFile == null ? getFilePath() : smlifFile;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		URI saveCurrentBaseURI = getCurrentBaseURI();			//	Alias processing may need access to the value, so save the incoming value
		super.endElement(uri,localName,qName);

		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			// </instances> element
			if (instanceElementHit && ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = false;
			}

			// </definitions> element
			else if (definitionElementHit && ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = false;
			}

			// </baseURI> element
			else if (baseURIElementHit && ISMLConstants.BASE_URI_ELEMENT.equals(localName))
			{
				baseURIElementHit = false;
			}

			// </alias> element
			else if (aliasElementHit && ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = false;
				int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
				IValidationOutput<?, ?> logger = getMessageOutputter(); 
				if (currentAlias.length() > 0)
				{
					//	if smlif:baseURI exists on a contained document, it is NOT used for aliases, per the SML-IF specification
					currentAlias = URIReference.validAliasOrAliasPrefix(currentAlias, saveCurrentBaseURI, 
							logger , lineNum , SMLValidationMessages.documentAlias);
					if (currentAlias == null)	
					{
						//	nothing to do 
					}
					//	valid so far (absolute form was producible, any fragment has been removed w/ warning msg)
					else if (allAliases.contains(currentAlias))
					{
						if (logger != null)
						{
							logger.reportMessage(ValidationMessageFactory.createErrorMessage(
									lineNum , NLS.bind(SMLValidationMessages.errorDuplicateAlias, currentAlias)));
						}
					}
					// Otherwise accept the alias
					else if (!currentAliases.contains(currentAlias))
					{
						currentAliases.add(currentAlias);
					}
				}
				currentAlias = "";
			}
			

			// </document> element
			else if (documentElementHit && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;
				allAliases.addAll(currentAliases);
				currentAliases.clear();
			}

			// </data> element
			else if (dataElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = false;

				if (currentElementModel != null)
				{
					String elementSource = currentElementModel.getSource();
					if (elementSource == null || elementSource.length() <= 0)	//	null unless this is locator/documenturi simulating data
					{
						processCurrentElementModel();	//	fills in source, sets aliases
					}
					else	//	locator/documenturi simulating data (see retrieveRemoteDocument where it walks the tree)
					{	
						currentElementModel.setAliases(currentAliases);
					}

					if(saveCurrentBaseURI != null)
					{
						currentElementModel.setDocumentBaseURI(saveCurrentBaseURI);
					}
					else if (!currentBaseURI.equals(""))	//	smlif:document/*/smlif:baseURI
					{	// TODO Check this code - this value should be in the stack if it was correct to use it, no?
						int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
						IValidationOutput<?, ?> logger = getMessageOutputter(); 
						URI candidateModelBaseURI = URIReference.validAliasOrAliasPrefixAsURI(currentBaseURI, saveCurrentBaseURI, logger, lineNum , SMLValidationMessages.documentAlias);
						if (candidateModelBaseURI != null)
						{
							currentElementModel.setDocumentBaseURI(candidateModelBaseURI);
						}
					}
					else	//	 No xml:base, no smlif:baseURI on document, use model smlif:baseURI (may also be null)
					{	
						SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
						if (identity != null )
						{
							String baseURI = identity.getBaseURI();
							if (baseURI != null) 
							{
								int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
								IValidationOutput<?, ?> logger = getMessageOutputter(); 
								URI candidateModelBaseURI = URIReference.validModelBaseURIAsURI(baseURI, null, logger, lineNum );
								if (candidateModelBaseURI != null)
								{
									currentElementModel.setDocumentBaseURI(candidateModelBaseURI);
								}
							}
						}
					}

					// Definition document
					if (definitionElementHit)
					{
						smlDocuments.addDefinition((ElementSchemaModel)currentElementModel);
					}

					// Instance document
					else if (instanceElementHit)
					{
						smlDocuments.addInstance(currentElementModel);
					}

					currentElementModel = null;
				}
			}

			// </base64Data> element
			else if (base64DataElementHit && ISMLConstants.BASE64DATA_ELEMENT.equals(localName)) {
				base64DataElementHit = false;

				if (currentElementModel != null) {
					currentElementModel.setAliases(currentAliases);

					if(saveCurrentBaseURI != null)
					{
						currentElementModel.setDocumentBaseURI(saveCurrentBaseURI);
					}
					else if (!currentBaseURI.equals(""))	//	smlif:document/*/smlif:baseURI
					{
						int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
						IValidationOutput<?, ?> logger = getMessageOutputter(); 
						URI candidateModelBaseURI = URIReference.validAliasOrAliasPrefixAsURI(currentBaseURI, saveCurrentBaseURI, logger, lineNum , SMLValidationMessages.documentAlias);
						if (candidateModelBaseURI != null)
						{
							currentElementModel.setDocumentBaseURI(candidateModelBaseURI);
						}
					}
					else	//	 No xml:base, no smlif:baseURI on document, use model smlif:baseURI (may also be null)
					{	
						SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
						if (identity != null )
						{
							String baseURI = identity.getBaseURI();
							if (baseURI != null) 
							{
								int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
								IValidationOutput<?, ?> logger = getMessageOutputter(); 
								URI candidateModelBaseURI = URIReference.validModelBaseURIAsURI(baseURI, null, logger, lineNum );
								if (candidateModelBaseURI != null)
								{
									currentElementModel.setDocumentBaseURI(candidateModelBaseURI);
								}
							}
						}
					}

					// Definition document
					if (definitionElementHit)
					{
						smlDocuments.addDefinition((ElementSchemaModel)currentElementModel);
					}

					// Instance document
					else if (instanceElementHit)
					{
						smlDocuments.addInstance(currentElementModel);
					}

					currentElementModel = null;
				}
			}
		}
	}


	private void processCurrentElementModel()
	{
		if (getLocator() != null)
		{
			currentElementModel.setEndingLine(getLocator().getLineNumber());
		}

		currentElementModel.setAliases(currentAliases);
		getElementSource(currentElementModel);
	}


	private void getElementSource(ElementModel elementModel)
	{
		LineNumberReader in = null;
		try {
			String path = inEclipse ? SMLValidatorUtil.retrieveAbsolutePath(elementModel.getFilePath()) : elementModel.getFilePath();
			in = new LineNumberReader(new FileReader(path));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		try {
			String currentLine = in.readLine();

			while (in.getLineNumber() < elementModel.getLineNumber()) {
				currentLine = in.readLine();
			}

			StringBuffer contents = new StringBuffer();
			if (smlifFile != null) {
				contents.append(ISMLConstants.XML_PREAMBLE + ISMLConstants.nl);
			}
			String firstLine = null;
			if (currentLine.indexOf('<') == -1) {
				firstLine = (currentLine = in.readLine()) + ISMLConstants.nl;
			} else {
				int prevTagOnLine = currentLine.lastIndexOf('>', elementModel.getColumnNumber());
				firstLine = currentLine.substring(prevTagOnLine + 1)+ISMLConstants.nl;
			}
			in.setLineNumber(elementModel.getLineNumber());
			contents.append(firstLine);
			boolean endFound = false;
			while (!endFound && (currentLine != null)) {
				currentLine = in.readLine();
				if (currentLine == null) {
					break;
				}
				// look for import element to tack SML schema location onto for validation
				if ((elementModel.getAdditionalSchemaLocation() != null) && (currentLine.matches(".*[^/][\\s]*"+ISMLConstants.IMPORT_ELEMENT+".*"))) {
					int endTagIndex = currentLine.indexOf('>', currentLine.indexOf(ISMLConstants.IMPORT_ELEMENT));
					if (endTagIndex == -1) {
						currentLine += elementModel.getAdditionalSchemaLocation();
					} else {
						if (currentLine.charAt(endTagIndex - 1) == '/') {
							// The import element is an empty element (i.e. the tag ends in "/>", so back up to the '/')
							endTagIndex--;
						}
						currentLine = currentLine.substring(0, endTagIndex) + elementModel.getAdditionalSchemaLocation() + currentLine.substring(endTagIndex);
					}
				}
				endFound = currentLine.matches(".*/[\\s]*"+elementModel.getElementName()+"[\\s]*>.*");
				if (endFound) {
					// this handles the case where there is more after the end tag of the element
					// for example, </xsi:schema> </data> appearing on the same line
					int endTagIndex = currentLine.indexOf(elementModel.getElementName());
					// find the first angle bracket immediately after the element name
					int rightBracket = currentLine.indexOf('>', endTagIndex);
					elementModel.setEndingLine(in.getLineNumber());
					elementModel.setEndingColumn(rightBracket);
					contents.append(currentLine.substring(0, rightBracket+1) + ISMLConstants.nl);
				} else {
					contents.append(currentLine + ISMLConstants.nl);
				}
			}
			elementModel.setSource(contents.toString());
			in.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}



	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public SMLDocuments getDataStructure()
	{
		return smlDocuments;
	}
}
