/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.net.URI;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This databuilder constructs an SMLIFIdentity object which will include
 * the content that is included in the identity section of an SML-IF
 * document.
 *
 * @author Ali Mehregani
 * @author John Arwe
 */
public class IdentityDataBuilder extends AbstractDataBuilder<SMLIFIdentity>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".IdentityDataBuilder";


	/**
	 * The sml-if identity object
	 */
	private SMLIFIdentity smlIfIdentity;

	/**
	 * Flags used to indicate the start of an element
	 */
	private boolean identityElementHit,					// Identity element hit
					nameElementHit,						// Name element hit
					descriptionElementHit,				// Description element hit
					baseURIHit;							// Base URI element hit

	/**
	 * The name stored in the identity section
	 */
	private String name;

	/**
	 * The description stored in the identity section
	 */
	private String description;

	/**
	 * The base URI stored in the smlif:model/smlif:identity section
	 */
	private String baseURI;


	/**
	 * Constructor
	 */
	public IdentityDataBuilder()
	{
		smlIfIdentity = new SMLIFIdentity();
		name = "";
		description = "";
		baseURI = "";
		super.addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		super.startElement(uri, localName, qName, attributes);
		elementNotification (uri, localName, attributes, true, getCurrentBaseURI());
	}


	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		if (nameElementHit)
		{
			name += new String(ch, start, length);
		}
		else if (descriptionElementHit)
		{
			description += new String(ch, start, length);
		}
		else if (baseURIHit)
		{
			baseURI += new String(ch, start, length);
		}
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		URI saveCurrentBaseURI = getCurrentBaseURI();
		super.endElement(uri, localName, qName);
		elementNotification (uri, localName, null, false, saveCurrentBaseURI);
	}

	private void validateBaseURI(URI currentBaseURI) {
		int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
		baseURI = baseURI.trim();
		if (URIReference.validModelBaseURI(baseURI, currentBaseURI, getMessageOutputter(), lineNum ))
		{
			smlIfIdentity.setBaseURI(baseURI);
		}
		// if it was not valid, the URIReference method logged messages already
		baseURI = "";
	}


	private void elementNotification (String uri, String localName, Attributes attributes, boolean start, URI currentBaseURI)
	{
		if (!ISMLConstants.SMLIF_URI.equals(uri))
			return;

		if (ISMLConstants.IDENTITY_ELEMENT.equals(localName)) {
			identityElementHit = start;
			if (!start && !("".equals(baseURI))) { // end of smlif:identity and a model base URI was specified via smlif:baseURI -or- model/@xml:base
				validateBaseURI(currentBaseURI);
			}
		}
		else if (identityElementHit)
		{
			if (ISMLConstants.NAME_ELEMENT.equals(localName))
			{
				nameElementHit = start;
				if (!start)
					smlIfIdentity.setName(name);
			}
			else if (ISMLConstants.DESCRIPTION_ELEMENT.equals(localName))
			{
				descriptionElementHit = start;
				if (!start)
					smlIfIdentity.setDescription(description);
			}
			else if (ISMLConstants.BASE_URI_ELEMENT.equals(localName))
			{
				baseURIHit = start;
			}
		}

	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public SMLIFIdentity getDataStructure()
	{
		return smlIfIdentity;
	}

	/**
	 * Retrieves and returns the base URI
	 * @return the base URI
	 */
	public static String retrieveBaseURI()
	{
		SMLIFIdentity smlIdentity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, ID);
		return smlIdentity == null ? null : smlIdentity.getBaseURI();
	}
}
