/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.export.ui;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.osgi.util.NLS;

/**
 * Content provider for displaying resources that can be selected for export
 * 
 * @author David Whiteman
 */
public class ExportResourceTreeContentProvider implements ITreeContentProvider {

	public ExportResourceTreeContentProvider() {
	}

	public Object[] getChildren(Object parentElement) {
		IResource[] kids = new IResource[0];
		if (parentElement instanceof IFile) {
			return kids;
		}
		try {
			kids = ((IContainer) parentElement).members();
		} catch (CoreException e) {
			e.printStackTrace();
		}
		if (kids.length == 0) return kids;
		return buildGoodKids(kids);
	}

	/*
	 *  Find all children that are definition or instance files
	 */
	protected Object[] buildGoodKids(IResource[] kids) {
		List<IResource> goodKids = new ArrayList<IResource>(kids.length);
		for (int i = 0; i < kids.length; i++) {
			IResource resource = getRefreshed(kids[i]);
			if (resource instanceof IContainer) {
				if (!resource.getName().startsWith(".")) {
					if ((resource instanceof IProject) || hasChildren(resource)) {
						goodKids.add(resource);
					}
				}
			} else {
				if (!resource.getName().startsWith(".")) 
				{					
					try
					{
						int type = SMLValidatorUtil.identifyDocumentType((IFile) resource);

						if ((type == ISMLConstants.TYPE_DEFINITION) || (type == ISMLConstants.TYPE_INSTANCE)) {
							goodKids.add(resource);
						}
					} catch (Exception e)
					{
						SMLCommonUtil.openErrorWithDetail(org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages.errorContentTypeTitle, 
								NLS.bind(org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages.errorContentType, ((IFile) resource).getFullPath()), e);
					} 
				}
			}
		}
		return goodKids.toArray();
	}

	/**
	 * Return the resource argument, first ensuring that it
	 * is refreshed in the workspace.
	 * 
	 * @param resource
	 * @return
	 */
	private IResource getRefreshed(IResource resource) {
		if (resource == null)
			return null;

		if (!resource.isSynchronized(IResource.DEPTH_ZERO)) {
			try {
				resource.refreshLocal(IResource.DEPTH_ZERO, null);
			} catch (CoreException e) {
				SMLCommonUtil.openErrorWithDetail(
						SMLMessages.errorSynchronizeTitle,
						SMLMessages.errorSynchronize, e, false);
				return null;
			}

		}
		return resource;
	}

	public Object getParent(Object element) {
		return ((IResource) element).getParent();
	}

	public boolean hasChildren(Object element) {
		IResource res = (IResource) element;
		return (res.getType() != IResource.FILE) && (getChildren(element).length > 0);
	}

	public Object[] getElements(Object inputElement) {
		return getChildren(inputElement);
	}

	public void dispose() {
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
	}
	
}
