/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.internal.cmdbf.services.query.transform.input.artifacts;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.artifacts.IContentSelector;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.artifacts.ITemplate;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IInstanceIdConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRecordConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IXPathExpression;

/**
 * Abstract class for all Templates
 * 
 * @author David Whiteman
 * @author Ali Mehregani
 */
public abstract class Template implements ITemplate, IXMLWritable 
{
	private String id;
	private Boolean suppressFromResult = null; // optional

	private IInstanceIdConstraint instanceIdConstraint; // optional
	private List<IXPathExpression> xpathExpressions = new ArrayList<IXPathExpression>(); // 0 or more
	private List<IRecordConstraint> recordConstraints = new ArrayList<IRecordConstraint>(); // 0 or more
	private IContentSelector contentSelector; // optional
	
	public Template(String id)
	{
		this.id = id;
	}
	
	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public boolean isSuppressFromResult() {
		if (primIsSuppressFromResult() == null) {
			return false;
		}
		return primIsSuppressFromResult().booleanValue();
	}

	private Boolean primIsSuppressFromResult() {
		return suppressFromResult;
	}

	public void setSuppressFromResult(boolean dropDirective) {
		this.suppressFromResult = Boolean.valueOf(dropDirective);
	}

	public IInstanceIdConstraint getInstanceIdConstraint() {
		return instanceIdConstraint;
	}

	public void setInstanceIdConstraint(IInstanceIdConstraint instanceIdConstraint) {
		this.instanceIdConstraint = instanceIdConstraint;
	}

	public IContentSelector getContentSelector() 
	{
		return contentSelector;
	}

	public void setContentSelector(IContentSelector contentSelector) 
	{
		this.contentSelector = contentSelector;
	}

	public IRecordConstraint[] getRecordConstraints() {
		return recordConstraints.toArray(new IRecordConstraint[recordConstraints.size()]);
	}

	public IXPathExpression[] getXpathExpressions() {
		return xpathExpressions.toArray(new IXPathExpression[xpathExpressions.size()]);
	}

	public void addXpathExpression(IXPathExpression expression) {
		xpathExpressions.add(expression);
	}
	
	public void addRecordConstraint(IRecordConstraint recordConstraint) {
		recordConstraints.add(recordConstraint);
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET+ICMDBfServicesConstants.CMDBF_PREFIX + getElementName()+ICMDBfServicesConstants.SINGLE_SPACE);
		CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.ID_ATTRIBUTE, getId());
		if (primIsSuppressFromResult() != null) {
			writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.SUPPRESS_FROM_RESULT_ATTRIBUTE, primIsSuppressFromResult().toString());
		}
		writer.write(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET + ICMDBfServicesConstants.nl);
		
		if (xpathExpressions.size() > 0) {
			// These are mutually exclusive with other constraints
			CMDBfServicesUtil.outputListToXML(writer, indent+1, xpathExpressions);
		} else {
			if (getContentSelector() != null) {
				getContentSelector().toXML(writer, indent+1);
			}
			if (getInstanceIdConstraint() != null) {
				getInstanceIdConstraint().toXML(writer, indent+1);
			}
			CMDBfServicesUtil.outputListToXML(writer, indent+1, recordConstraints);
		}

		outputAdditionalSubelements(writer, indent+1);

		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + getElementName())+ICMDBfServicesConstants.nl);
	}

	protected void outputAdditionalSubelements(StringWriter writer, int indent) {
	}

	protected abstract String getElementName();

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((contentSelector == null) ? 0 : contentSelector.hashCode());
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime
				* result
				+ ((instanceIdConstraint == null) ? 0 : instanceIdConstraint
						.hashCode());
		result = prime
				* result
				+ ((recordConstraints == null) ? 0 : recordConstraints
						.hashCode());
		result = prime
				* result
				+ ((suppressFromResult == null) ? 0 : suppressFromResult
						.hashCode());
		result = prime
				* result
				+ ((xpathExpressions == null) ? 0 : xpathExpressions.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final Template other = (Template) obj;
		if (contentSelector == null) {
			if (other.contentSelector != null)
				return false;
		} else if (!contentSelector.equals(other.contentSelector))
			return false;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		if (instanceIdConstraint == null) {
			if (other.instanceIdConstraint != null)
				return false;
		} else if (!instanceIdConstraint.equals(other.instanceIdConstraint))
			return false;
		if (recordConstraints == null) {
			if (other.recordConstraints != null)
				return false;
		} else if (!recordConstraints.equals(other.recordConstraints))
			return false;
		if (suppressFromResult == null) {
			if (other.suppressFromResult != null)
				return false;
		} else if (!suppressFromResult.equals(other.suppressFromResult))
			return false;
		if (xpathExpressions == null) {
			if (other.xpathExpressions != null)
				return false;
		} else if (!xpathExpressions.equals(other.xpathExpressions))
			return false;
		return true;
	}
}
