/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations.cmdbf;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfMessages;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRecordType;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileInstanceDocument;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.w3c.dom.Node;

/**
 * An implementation of the record type constraint handler.  This constraint passes
 * if the root element of a given SML document satisfies one of the following
 * conditions:
 * <ul>
 *  <li>recordType/@namespace = root/@namespace && recordType/@localName = root/@localName </li>
 *  <li>type(root) is derived from type(recordType/@namespace, root/@localName) </li>
 * </ul>
 * 
 * @author Ali Mehregani
 */
public class ItemRecordHandler extends AbstractSMLConstraintHandler
{
	/** 
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.impl.AbstractItemConstraintHandler#handle(org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.INodes, org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	@SuppressWarnings("unchecked")
	protected INodes handle(INodes context, IConstraint constraint) throws CMDBfServiceException
	{				
		try
		{
			INodes results = retrieveContext(context);
			IGraphElement[] elements = results.getElements();
			
			ISMLRepository repository = getRepository();
			IRecordType recordType = (IRecordType)constraint;
			String namespace = CMDBfServicesUtil.toString(recordType.getNamespace());
			String localName = recordType.getLocalName();
			List<IGraphElement> revisedList = new ArrayList<IGraphElement>();
			for (int i = 0; i < elements.length; i++)
			{
				SMLFileInstanceDocument document = CMDBfUtil.retrieveDocument(repository, (IItem)elements[i]);			
				Node rootNode = document == null ? null : document.getDOMDocument().getFirstChild();
				
				// Continue if the root element node cannot be resolved
				if (rootNode == null)
				{				
					continue;
				}
							
				// This constraint passes if the root element of the SML document
				// has the same name space and local name as the record type constraint
				if (namespace.equals(rootNode.getNamespaceURI()) && localName.equals(rootNode.getLocalName()))
				{
					revisedList.add(elements[i]);
					continue;
				}
				
				// The constraint will also pass if the root node happens to be a derived type
				// of the element with the name space and local name specified in the record type constraint			
				SMLRepositoryUtil.parseDocument(document, repository);
				Map<String, Map<String, String>> typeInfo = (Map<String, Map<String, String>>)repository.getProperty(IFileSystemSMLProperties.TYPE_INFORMATION, (Object)null);
				String type = SMLRepositoryUtil.retrieveType(typeInfo, namespace, localName);
				int colonInx;
				if (type != null && (colonInx = type.lastIndexOf(':')) >= 0)
				{				
					String typeNamespace = type.substring(0, colonInx);
					String typeLocalName = type.substring(colonInx + 1);

					if (SMLRepositoryUtil.isDerivedType(repository, rootNode, typeNamespace, typeLocalName))
					{
						revisedList.add(elements[i]);
						continue;
					}				
				}
			}
			
			results.setElements(revisedList.toArray(new IGraphElement[revisedList.size()]));
			return results;
		} 
		catch (RepositoryOperationException e)
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(e.getLocalizedMessage()),
					e);			
		}
	}
}
